#pragma once

#include <rtline/util/types/interval.h>

#include <library/cpp/json/json_value.h>

#include <util/datetime/base.h>
#include <util/generic/maybe.h>

namespace NDrive {
    class TStatus {
    public:
        TStatus() = default;
        TStatus(const NJson::TJsonValue& json);
        TStatus(const TString& status, const TString& actionId, const TString& carId, const TString& sessionId, const TString& userId, TInstant start)
            : CarStatus(status)
            , Start(start)
            , ActionId(actionId)
            , CarId(carId)
            , SessionId(sessionId)
            , UserId(userId)
        {
        }

        TInterval<ui64> GetInterval() const;

        const TString& GetCarStatus() const {
            return CarStatus;
        }
        TInstant GetStart() const {
            return Start;
        }
        TInstant GetFinish() const {
            return Finish.GetRef();
        }
        TMaybe<ui64> GetIMEI() const {
            return IMEI;
        }
        ui64 GetIndex() const {
            return Index;
        }

        const TString& GetActionId() const {
            return ActionId;
        }
        const TString& GetCarId() const {
            return CarId;
        }
        const TString& GetSessionId() const {
            return SessionId;
        }
        const TString& GetUserId() const {
            return UserId;
        }
        const TString& GetActionPrefix() const {
            return ActionPrefix;
        }
        const TString& GetActionSuffix() const {
            return ActionSuffix;
        }

        bool IsCurrent() const {
            return Finish.Empty();
        }

        void SetCarStatus(const TString& value) {
            CarStatus = value;
        }
        void SetActionPrefix(const TString& value) {
            ActionPrefix = value;
        }
        void SetActionSuffix(const TString& value) {
            ActionSuffix = value;
        }
        void SetActionId(const TString& value) {
            ActionId = value;
        }
        void SetUserId(const TString& value) {
            UserId = value;
        }
        void SetIMEI(ui64 value) {
            IMEI = value;
        }
        void SetIndex(ui64 value) {
            Index = value;
        }
        void SetStart(TInstant value) {
            Start = value;
        }
        void SetFinish(TInstant value) {
            Y_ENSURE(value >= Start);
            Finish = value;
        }

        bool operator==(const TStatus& other) const;
        bool operator<(const TStatus& other) const;
        bool operator<(TInstant timestamp) const;

        Y_SAVELOAD_DEFINE(
            CarStatus,
            Start,
            Finish,
            IMEI,
            Index,
            ActionId,
            CarId,
            SessionId,
            UserId,
            ActionPrefix,
            ActionSuffix
        );

    public:
        static TStatus Incorrect(ui64 imei = 0, const TString& carId = {}, ui64 index = 0);
        static TStatus Zero(ui64 imei = 0, const TString& carId = "null", ui64 index = 0);

    private:
        TString CarStatus;
        TInstant Start;
        TMaybe<TInstant> Finish;
        TMaybe<ui64> IMEI;
        ui64 Index = 0;

        TString ActionId;
        TString CarId;
        TString SessionId;
        TString UserId;
        TString ActionPrefix;
        TString ActionSuffix;
    };
}
