#include "timeline.h"

#include <drive/library/cpp/threading/future.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/threading/future/async.h>

#include <rtline/library/unistat/signals.h>
#include <rtline/util/algorithm/container.h>
#include <rtline/util/algorithm/ptr.h>

namespace {
    class TBoolGuard {
    public:
        TBoolGuard(bool& value) noexcept
            : Value(value)
        {
            Value = true;
        }
        ~TBoolGuard() noexcept {
            Value = false;
        }

    private:
        bool& Value;
    };

    TUnistatSignal<> TimelineGlobalLag({ "timeline-global-lag" }, EAggregationType::LastValue, "axxx");
    TUnistatSignal<> TimelineGlobalUpdate({ "timeline-global-update" }, false);
    TUnistatSignal<> TimelineLocalUpdate({ "timeline-local-update" }, false);
}

NDrive::TNewTimeline::TNewTimeline(TRequesterPtr requester, ui64 imei, TDuration depth, TDuration updateInterval, TDuration realtimePause)
    : TBaseTimeline(imei, updateInterval)
    , Requester(requester)
    , RealtimePause(realtimePause)
{
    auto now = Now();
    bool exists = true;
    Y_ENSURE(Requester, "requester is missing on creating timeline for " << imei);
    {
        auto match = Requester->GetCars({ imei });
        auto p = match.find(imei);
        if (p != match.end()) {
            Car = p->second;
            exists = true;
        } else {
            exists = false;
        }
    }
    INFO_LOG << "IMEI " << GetIMEI() << ": creating new timeline " << Car.Index << ":" << Car.Id << " " << Car.Model << " from scratch" << Endl;
    if (!exists) {
        Y_ENSURE(AddStatus(TStatus::Incorrect(imei), now));
        return;
    } else {
        Y_ENSURE(AddStatus(TStatus::Zero(imei, Car.Id, Car.Index), now));
    }

    LowTimestampLimit = now - depth;
    Update(LowTimestampLimit, now);
}

NDrive::TNewTimeline::TNewTimeline(TRequesterPtr requester, const TSessionRequester::TCar& car, TDuration depth, TDuration updateInterval, TDuration realtimePause)
    : TBaseTimeline(car.IMEI, updateInterval)
    , Requester(requester)
    , Car(car)
    , RealtimePause(realtimePause)
{
    auto now = Now();

    INFO_LOG << "IMEI " << GetIMEI() << ": creating new timeline " << Car.Index << ":" << Car.Id << " " << Car.Model << " from car" << Endl;
    Y_ENSURE(AddStatus(TStatus::Zero(Car.IMEI, Car.Id, Car.Index), now));
    LowTimestampLimit = now - depth;
    Update(LowTimestampLimit, now);
}

bool NDrive::TNewTimeline::AddStatus(TStatus&& status, TInstant timestamp) {
    if (status.GetIndex() != Car.Index) {
        return false;
    }
    return TBaseTimeline::AddStatus(std::move(status), timestamp);
}

TString NDrive::TNewTimeline::GetNativeStatus() const {
    TReadGuard guard(Lock);
    return Car.Status;
}

void NDrive::TNewTimeline::SetNativeStatus(const TString& value) {
    TWriteGuard guard(Lock);
    Car.Status = value;
}

NDrive::TStatus NDrive::TNewTimeline::OnMissingStatus(TMaybe<TStatus> previous, TInstant timestamp, ui32 attempt) {
    auto last = std::max(GetUpdateTimestamp(), LowTimestampLimit);
    auto now = Now();
    if (timestamp > now) {
        ythrow yexception() << "data from the future: " << timestamp;
    }
    if (timestamp > last + GetUpdateInterval()) {
        Update(last, now);
        return GetStatusImpl(timestamp, attempt + 1);
    } else {
        TInstant start = previous ? previous->GetFinish() : TInstant::Zero();
        TString carId = previous ? previous->GetCarId() : "unknown_car_id";
        TString actionId = ToString(start.Seconds()) + "-" + carId;
        NDrive::TStatus status("available", actionId, carId, "", "", start);
        return status;
    }
}

void NDrive::TNewTimeline::Update(TInstant since, TInstant until) {
    CHECK_WITH_LOG(until >= since);
    if (until == since) {
        return;
    }
    if (RealtimePause) {
        until -= RealtimePause;
    }

    Y_ENSURE(Requester, "requester is missing to update timeline " << GetIMEI());
    auto statuses = Requester->GetDelta(since, until, NContainer::Scalar(Car), true);
    TimelineLocalUpdate.Signal(1);
    for (auto&& status : statuses) {
        status.SetIMEI(GetIMEI());
        AddStatus(std::move(status), until);
    }
    AddTimestamp(until);
}

TVector<NDrive::TStatus> NDrive::TBaseTimeline::GetAllStatuses() {
    TReadGuard guard(Lock);
    return Statuses;
}

NDrive::TStatus NDrive::TBaseTimeline::GetCurrentStatus() {
    return GetStatusImpl(Now(), 0);
}

NDrive::TStatus NDrive::TBaseTimeline::GetStatus(TInstant timestamp) {
    return GetStatusImpl(timestamp, 0);
}

NDrive::TStatus NDrive::TBaseTimeline::GetPreviousStatus(const TStatus& status) {
    return GetStatus(status.GetStart() - TDuration::Seconds(1));
}

NDrive::TStatus NDrive::TBaseTimeline::GetLastActionableStatus(const TStatus& status, ui32& hops) {
    hops = 0;
    auto result = status;
    while (!result.GetActionId() && result.GetStart() > TInstant::Zero()) {
        result = GetPreviousStatus(result);
        hops++;
    }
    return result;
}

void NDrive::TBaseTimeline::AddTimestamp(TInstant timestamp) {
    UpdateTimestamp = std::max(UpdateTimestamp, timestamp);
}

NDrive::TStatus NDrive::TBaseTimeline::GetStatusImpl(TInstant timestamp, ui32 attempt) {
    TMaybe<NDrive::TStatus> previous;
    {
        TReadGuard guard(Lock);
        CHECK_WITH_LOG(attempt < 2);
        Y_ASSERT(!Statuses.empty());
        Y_ASSERT(std::is_sorted(Statuses.begin(), Statuses.end()));
        auto p = std::lower_bound(Statuses.begin(), Statuses.end(), timestamp + TDuration::MicroSeconds(1));
        p = std::max(p - 1, Statuses.begin());
        const NDrive::TStatus& status = *p;
        if (status.IsCurrent()) {
            if (timestamp < UpdateTimestamp + UpdateInterval) {
                return status;
            }
        } else {
            if (status.GetFinish() > timestamp) {
                return status;
            } else {
                previous = status;
            }
        }
    }
    return OnMissingStatus(previous, timestamp, attempt);
}

bool NDrive::TBaseTimeline::AddStatus(TStatus&& status, TInstant timestamp) {
    TWriteGuard guard(Lock);
    AddTimestamp(timestamp);
    Y_ASSERT(std::is_sorted(Statuses.begin(), Statuses.end()));
    if (std::binary_search(Statuses.begin(), Statuses.end(), status)) {
        DEBUG_LOG << "IMEI " << IMEI << ": duplicate status " << status << Endl;
        return false;
    } else {
        DEBUG_LOG << "IMEI " << IMEI << ": adding status " << status << Endl;
        bool shouldSort = !Statuses.empty() && status < Statuses.back();
        Statuses.push_back(std::move(status));
        if (shouldSort) {
            std::stable_sort(Statuses.begin(), Statuses.end());
        }
        return true;
    }
}

void NDrive::TBaseTimeline::BumpStatus(TInstant timestamp) {
    AddTimestamp(timestamp);
}

NDrive::TNewTimelines::TNewTimelines(TRequesterPtr requester, const TOptions& options /*= Default<TOptions>()*/)
    : IAutoActualization("NewTimelines", options.UpdateInterval)
    , Requester(requester)
    , Options(options)
    , Updating(false)
{
    Pool.Start(32);
    Update();
    Y_ENSURE_BT(Start());
}

NDrive::TNewTimelines::~TNewTimelines() {
    if (!Stop()) {
        ERROR_LOG << "cannot stop NewTimelines" << Endl;
    }
    Pool.Stop();
}

void NDrive::TNewTimelines::Preload() {
    if (Options.Preload) {
        INFO_LOG << "Preloading" << Endl;
        Y_ENSURE(Requester, "requester is missing to preload timelines");
        auto cars = Requester->GetCars();
        NThreading::TFutures<TAtomicSharedPtr<NDrive::TNewTimeline>> timelines;
        for (auto&& i : cars) {
            const auto& car = i.second;
            const ui64 imei = car.IMEI;
            if (!imei) {
                continue;
            }

            timelines.push_back(NThreading::Async([this, car] {
                return CreateTimeline(car);
            }, Pool));
        }
        for (auto&& i : timelines) {
            i.Wait();
            if (i.HasValue()) {
                auto timeline = i.ExtractValue();
                const TString& carId = timeline->GetCar().Id;
                const ui64 imei = timeline->GetCar().IMEI;
                Timelines.emplace(imei, timeline);
                CarId2IMEI[carId] = imei;
            } else if (i.HasException()) {
                ERROR_LOG << "an exception occurred: " << NThreading::GetExceptionMessage(i) << Endl;
            }
        }
        INFO_LOG << "Preloaded " << cars.size() << " objects " << Endl;
    }
}

TAtomicSharedPtr<NDrive::TNewTimeline> NDrive::TNewTimelines::AddTimeline(TNewTimeline&& timeline) {
    TWriteGuard guard(Lock);
    auto imei = timeline.GetIMEI();
    auto result = MakeAtomicShared<NDrive::TNewTimeline>(std::move(timeline));
    Timelines[imei] = result;
    return result;
}

template <class T>
TAtomicSharedPtr<NDrive::TNewTimeline> NDrive::TNewTimelines::CreateTimeline(const T& from) const {
    return MakeAtomicShared<NDrive::TNewTimeline>(Requester, from, Options.UpdateLocalDepth, Options.UpdateLocalInterval, Options.UpdateRealtimePause);
}

TAtomicSharedPtr<NDrive::TNewTimeline> NDrive::TNewTimelines::GetTimeline(ui64 imei) {
    if (Options.EnableGlobalUpdates) {
        Update();
    }
    return GetOrCreateTimelineImpl(imei);
}

TAtomicSharedPtr<NDrive::TNewTimeline> NDrive::TNewTimelines::GetTimelineImpl(ui64 imei) {
    {
        TReadGuard guard(Lock);
        auto p = Timelines.find(imei);
        if (p != Timelines.end()) {
            return p->second;
        }
    }
    return nullptr;
}

TAtomicSharedPtr<NDrive::TNewTimeline> NDrive::TNewTimelines::GetOrCreateTimelineImpl(ui64 imei) {
    auto existing = GetTimelineImpl(imei);
    if (existing) {
        return existing;
    }
    {
        TWriteGuard guard(Lock);
        auto p = Timelines.find(imei);
        if (p != Timelines.end()) {
            return p->second;
        } else {
            auto timeline = CreateTimeline(imei);
            return Timelines.emplace(imei, timeline).first->second;
        }
    }
}

TVector<NDrive::TSessionRequester::TCar> NDrive::TNewTimelines::GetCars() {
    TReadGuard guard(Lock);
    TVector<NDrive::TSessionRequester::TCar> result;
    for (auto&&[discarded, timeline] : Timelines) {
        if (timeline) {
            result.push_back(timeline->GetCar());
        }
    }
    return result;
}

ui64 NDrive::TNewTimelines::GetCount() const {
    TReadGuard guard(Lock);
    return Timelines.size();
}

void NDrive::TNewTimelines::Update() {
    if (!Requester) {
        ERROR_LOG << "Requester is missing" << Endl;
        return;
    }
    if (!Options.EnableGlobalUpdates) {
        INFO_LOG << "Global updates are disabled" << Endl;
        return;
    }

    auto now = Now() - Options.UpdateRealtimePause;
    auto lag = now - UpdateTimestamp;
    TimelineGlobalLag.Signal(lag.Seconds());

    if (Updating) {
        if (lag < Options.UpdatePauseLag) {
            return;
        } else {
            auto guard = Guard(UpdateMutex);
        }
    }

    TTryGuard<TMutex> guard(UpdateMutex);
    if (!guard.WasAcquired()) {
        return;
    }
    TBoolGuard boolGuard(Updating);

    if (now < UpdateTimestamp + Options.UpdateInterval) {
        return;
    }
    if (now > UpdateTimestamp + Options.UpdateGlobalDepth) {
        TWriteGuard guard(Lock);
        Timelines.clear();
        Preload();
        UpdateTimestamp = now;
    }
    if (now > UpdateCarsTimestamp + Options.UpdateCarsInterval) try {
        INFO_LOG << "Updating cars" << Endl;
        auto cars = Requester->GetCars();
        auto timelines = TMap<TString, NThreading::TFuture<TAtomicSharedPtr<NDrive::TNewTimeline>>>();
        ui32 updated = 0;
        for (auto&& [id, car] : cars) {
            if (!car.IMEI) {
                DEBUG_LOG << "Skip object without IMEI " << car.Id << Endl;
                continue;
            }
            auto existing = GetTimelineImpl(car.IMEI);
            if (!existing || existing->GetCar() != car) {
                timelines.emplace(car.Id, NThreading::Async([this, car = car] {
                    INFO_LOG << "Updating " << car.Id << " " << car.IMEI << Endl;
                    return CreateTimeline(car);
                    INFO_LOG << "Updated " << car.Id << " " << car.IMEI << Endl;
                }, Pool));
            }
            if (existing && existing->GetNativeStatus() != car.Status) {
                INFO_LOG << "Updating " << car.Id << " status from " << existing->GetNativeStatus() << " to " << car.Status << Endl;
                existing->SetNativeStatus(car.Status);
            }
        }
        for (auto&& [id, asyncTimeline] : timelines) {
            INFO_LOG << "Applying update for " << id << Endl;
            auto timeline = asyncTimeline.ExtractValueSync();
            auto imei = Yensured(timeline)->GetIMEI();
            TWriteGuard guard(Lock);
            Timelines[imei] = timeline;
            updated += 1;
        }
        INFO_LOG << "Updated " << updated << " cars" << Endl;
        UpdateCarsTimestamp = now;
    } catch (const std::exception& e) {
        ERROR_LOG << "Cannot update cars: " << FormatExc(e) << Endl;
    }

    auto timestamp = std::max(UpdateTimestamp - Options.UpdateOverlap, now - Options.UpdateGlobalDepth);
    INFO_LOG << "Updating new timelines: " << timestamp.SecondsFloat() << Endl;

    auto statuses = Requester->GetDelta(timestamp, now);
    TimelineGlobalUpdate.Signal(1);

    ui32 added = 0;
    for (auto&& status : statuses) {
        const TString& carId = status.GetCarId();
        if (!status.GetIMEI()) {
            auto p = CarId2IMEI.find(carId);
            if (p == CarId2IMEI.end()) {
                auto carId2IMEI = Requester->GetCars({ carId });
                auto i = carId2IMEI.find(carId);
                if (i != carId2IMEI.end()) {
                    auto imei = i->second.IMEI;
                    if (imei) {
                        p = CarId2IMEI.emplace(i->first, imei).first;
                    }
                }
            }
            if (p != CarId2IMEI.end()) {
                status.SetIMEI(p->second);
            }
        }
        if (status.GetIMEI() && *status.GetIMEI()) {
            auto timeline = GetOrCreateTimelineImpl(*status.GetIMEI());
            auto result = timeline->AddStatus(std::move(status), now);
            added += result ? 1 : 0;
        } else {
            ERROR_LOG << "Status without IMEI: " << status << Endl;
        }
    }

    TTimelinesContainer timelines;
    {
        TReadGuard rg(Lock);
        timelines = Timelines;
    }
    for (auto&&[imei, timeline] : timelines) {
        timeline->BumpStatus(now);
    }

    UpdateTimestamp = now;
    INFO_LOG << "Updated timelines: " << statuses.size() << " statuses acquired, " << added << " added, " << Timelines.size() << " IMEIs total" << Endl;
}

bool NDrive::TNewTimelines::Refresh() {
    Update();
    return true;
}
