#pragma once

#include "status.h"
#include "requester.h"

#include <rtline/util/auto_actualization.h>

#include <library/cpp/json/json_value.h>

#include <util/system/mutex.h>
#include <util/system/rwlock.h>

namespace NDrive {
    class TBaseTimeline {
    public:
        TBaseTimeline()
            : IMEI(0)
        {
        }
        TBaseTimeline(ui64 imei, TDuration updateInterval)
            : IMEI(imei)
            , UpdateInterval(updateInterval)
        {
        }
        TBaseTimeline(const TBaseTimeline& other)
            : IMEI(other.IMEI)
            , UpdateInterval(other.UpdateInterval)
            , Statuses(other.Statuses)
            , UpdateTimestamp(other.UpdateTimestamp)
        {
        }
        virtual ~TBaseTimeline() = default;

        ui64 GetIMEI() const {
            return IMEI;
        }
        TInstant GetUpdateTimestamp() const {
            return UpdateTimestamp;
        }

        virtual bool AddStatus(TStatus&& status, TInstant timestamp);
        virtual void BumpStatus(TInstant timestamp);

        TVector<TStatus> GetAllStatuses();
        TStatus GetCurrentStatus();
        TStatus GetStatus(TInstant timestamp);
        TStatus GetPreviousStatus(const TStatus& status);
        TStatus GetLastActionableStatus(const TStatus& status, ui32& hops);

    protected:
        virtual TStatus OnMissingStatus(TMaybe<TStatus> previous, TInstant timestamp, ui32 attempt) = 0;

    protected:
        void AddTimestamp(TInstant timestamp);
        TStatus GetStatusImpl(TInstant timestamp, ui32 attempt);

        TDuration GetUpdateInterval() const {
            return UpdateInterval;
        }

    protected:
        ui64 IMEI;
        TDuration UpdateInterval;

        TVector<TStatus> Statuses;
        TInstant UpdateTimestamp;

        TRWMutex Lock;
    };

    class TNewTimeline: public TBaseTimeline {
    public:
        TNewTimeline() = default;
        TNewTimeline(
            TRequesterPtr requester,
            ui64 imei,
            TDuration depth = TDuration::Days(3),
            TDuration updateInterval = TDuration::Seconds(30),
            TDuration realtimePause = TDuration::Zero()
        );
        TNewTimeline(
            TRequesterPtr requester,
            const TSessionRequester::TCar& car,
            TDuration depth = TDuration::Days(3),
            TDuration updateInterval = TDuration::Seconds(30),
            TDuration realtimePause = TDuration::Zero()
        );

        const TSessionRequester::TCar& GetCar() const {
            return Car;
        }

        bool AddStatus(TStatus&& status, TInstant timestamp) override;

        TString GetNativeStatus() const;
        void SetNativeStatus(const TString& value);

        Y_SAVELOAD_DEFINE(
            IMEI,
            UpdateTimestamp,
            Statuses,
            UpdateTimestamp,
            Car,
            LowTimestampLimit,
            RealtimePause
        );

    protected:
        virtual TStatus OnMissingStatus(TMaybe<TStatus> previous, TInstant timestamp, ui32 attempt) override;

    private:
        void Update(TInstant since, TInstant until);

    private:
        TRequesterPtr Requester;
        TSessionRequester::TCar Car;
        TInstant LowTimestampLimit;
        TDuration RealtimePause;
    };

    class TNewTimelines: protected IAutoActualization {
    public:
        struct TOptions {
            TDuration UpdateInterval = TDuration::Seconds(10);
            TDuration UpdateLocalInterval = TDuration::Seconds(30);
            TDuration UpdateLocalDepth = TDuration::Hours(37);
            TDuration UpdateGlobalDepth = TDuration::Minutes(90);
            TDuration UpdateOverlap = TDuration::Minutes(1);
            TDuration UpdatePauseLag = TDuration::Minutes(3);
            TDuration UpdateCarsInterval = TDuration::Minutes(10);
            TDuration UpdateRealtimePause = TDuration::Seconds(10);
            bool EnableGlobalUpdates = true;
            bool Preload = true;
        };

    public:
        TNewTimelines(TRequesterPtr requester, const TOptions& options = Default<TOptions>());
        ~TNewTimelines();

        TAtomicSharedPtr<TNewTimeline> AddTimeline(TNewTimeline&& timeline);
        TAtomicSharedPtr<TNewTimeline> GetTimeline(ui64 imei);
        TVector<TSessionRequester::TCar> GetCars();
        ui64 GetCount() const;

    private:
        using TTimelinesContainer = TMap<ui64, TAtomicSharedPtr<TNewTimeline>>;

    private:
        TAtomicSharedPtr<TNewTimeline> GetOrCreateTimelineImpl(ui64 imei);
        TAtomicSharedPtr<TNewTimeline> GetTimelineImpl(ui64 imei);

        template <class T>
        TAtomicSharedPtr<TNewTimeline> CreateTimeline(const T& from) const;

        void Preload();
        void Update();

        virtual bool Refresh() override;

    private:
        const TRequesterPtr Requester;
        const TOptions Options;

        TThreadPool Pool;
        TTimelinesContainer Timelines;
        TRWMutex Lock;
        TInstant UpdateTimestamp;
        TInstant UpdateCarsTimestamp;
        TMutex UpdateMutex;
        TMap<TString, ui64> CarId2IMEI;
        bool Updating;
    };
}
