#include "client.h"
#include <kernel/daemon/config/daemon_config.h>
#include <util/stream/file.h>

bool TTolokaClient::OpenPool(const TString& poolId) const {
    NNeh::THttpRequest httpRequest = BuildBaseRequest(TOpenPoolRequest::GetPath(poolId)).SetPostData(TOpenPoolRequest::GetJsonPost());

    NUtil::THttpReply reply = ClientSender->SendMessageSync(httpRequest);

    NJson::TJsonValue parsedReply;
    if (reply.Code() != HTTP_ACCEPTED || !ReadJsonFastTree(reply.Content(), &parsedReply)) {
        ERROR_LOG << reply.Code() << "/" << reply.Content() << Endl;
        return false;
    }
    return true;
}

bool TTolokaClient::CreatePool(const TString& poolName, const TString& description, TString& newPoolId, const NJson::TJsonValue& qualityControl) const {
    auto postData = TCreatePoolRequest::GetJsonPost(Config.GetProjectId(), poolName, description, Now() + TDuration::Days(120), qualityControl);
    NNeh::THttpRequest httpRequest = BuildBaseRequest(TCreatePoolRequest::GetPath()).SetPostData(postData);
    NUtil::THttpReply reply = ClientSender->SendMessageSync(httpRequest);

    NJson::TJsonValue parsedReply;
    if (reply.Code() != HTTP_CREATED || !ReadJsonFastTree(reply.Content(), &parsedReply)) {
        ERROR_LOG << reply.Code() << "/" << reply.Content() << Endl;
        return false;
    }
    newPoolId = parsedReply["id"].GetStringRobust();
    return !newPoolId.empty();
}

bool TTolokaClient::GetPoolInfo(const TString& poolId, bool& isOpened) const {
    NNeh::THttpRequest httpRequest = BuildBaseRequest(TGetPoolInfoRequest::GetPath(poolId));
    NUtil::THttpReply reply = ClientSender->SendMessageSync(httpRequest);

    NJson::TJsonValue parsedReply;
    if (reply.Code() != HTTP_OK || !ReadJsonFastTree(reply.Content(), &parsedReply)) {
        ERROR_LOG << reply.Code() << "/" << reply.Content() << Endl;
        return false;
    }

    const TString poolStatus = parsedReply["status"].GetString();
    const TString closingReason = parsedReply["last_close_reason"].GetString();
    isOpened = (poolStatus == "CLOSED" && closingReason == "COMPLETED");
    return true;
}


void TTolokaClientConfig::Init(const TYandexConfig::Section* section) {
    TBase::Init<TTolokaClientDefaults>(section);
    Token = GetEnv("TOLOKA_TOKEN", "");
    AssertCorrectConfig(!!Token, "Can't find Toloka Token");

    ProjectId = section->GetDirectives().Value("ProjectId", ProjectId);
    AssertCorrectConfig(!!ProjectId, "Can't find ProjectId");

    TasksInSet = section->GetDirectives().Value("TasksInSet", TasksInSet);
    QualityControlConfigPath = section->GetDirectives().Value("QualityControlConfigPath", QualityControlConfigPath);

    AssertCorrectConfig(!!QualityControlConfigPath, "Can't find QualityControlConfig file");
    TFileInput input(QualityControlConfigPath);
    AssertCorrectConfig(ReadJsonFastTree(input.ReadAll(), &QualityControlConfig), "Can't parse QualityControlConfig");
}


void TTolokaClientConfig::ToString(IOutputStream& os) const {
    TBase::ToString(os);
    os << "ProjectId: " << ProjectId << Endl;
    os << "QualityControlConfigPath: " << QualityControlConfigPath << Endl;
    os << "TasksInSet: " << TasksInSet << Endl;
}

TTolokaClientConfig TTolokaClientConfig::ParseFromString(const TString& configStr) {
    TTolokaClientConfig result;
    TAnyYandexConfig config;
    CHECK_WITH_LOG(config.ParseMemory(configStr.data()));
    result.Init(config.GetRootSection());
    return result;
}
