#pragma once

#include <rtline/util/network/simple.h>
#include <rtline/util/types/accessor.h>

#include <library/cpp/cgiparam/cgiparam.h>
#include <library/cpp/http/misc/httpcodes.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/json_value.h>
#include <library/cpp/logger/global/global.h>
#include <library/cpp/mediator/global_notifications/system_status.h>
#include <library/cpp/yconf/conf.h>

#include <util/datetime/base.h>
#include <util/system/env.h>

struct TTolokaClientDefaults: public TSimpleAsyncRequestSender::TConfigDefaults {
    static TString GetHost() {
        return "toloka.yandex.ru";
    }
    static ui32 GetPort() {
        return 443;
    }
    static bool GetIsHttps() {
        return true;
    }
    static TDuration GetGlobalTimeout() {
        return TDuration::Seconds(5);
    }
    static ui32 GetMaxAttempts() {
        return 1;
    }
};

class TTolokaClientConfig: public TSimpleAsyncRequestSender::TConfig {
private:
    using TBase = TSimpleAsyncRequestSender::TConfig;
    R_READONLY(TString, Token);
    R_READONLY(TString, ProjectId);
    R_READONLY(TString, QualityControlConfigPath);
    R_READONLY(ui32, TasksInSet, 15);
    R_READONLY(NJson::TJsonValue, QualityControlConfig, NJson::TJsonValue(NJson::JSON_MAP));

public:
    void Init(const TYandexConfig::Section* section);

    void ToString(IOutputStream& os) const;

    static TTolokaClientConfig ParseFromString(const TString& configStr);
};

class TTolokaClient {
public:
    TTolokaClient(const TTolokaClientConfig& config)
        : Config(config)
        , ClientSender(config)
    {}

public:
    bool CreatePool(const TString& poolName, const TString& description, TString& newPoolId, const NJson::TJsonValue& qualityControl) const;

    NNeh::THttpRequest BuildBaseRequest(const TString& uri) const {
        NNeh::THttpRequest result;
        result.SetOAuthToken(Config.GetToken()).SetContentType("application/json").SetUri(uri);
        return result;
    }

    template <class TTaskType>
    bool AddTasks(const TString& poolId, const TVector<TTaskType>& tasks, ui32 attempt = 3) const {
        if (attempt == 0) {
            return false;
        }
        NNeh::THttpRequest httpRequest = BuildBaseRequest(TCreateSuiteRequest::GetPath());
        httpRequest.SetPostData(TCreateSuiteRequest::GetJsonPost(poolId, tasks));

        NUtil::THttpReply reply = ClientSender->SendMessageSync(httpRequest);

        if (reply.Code() != HTTP_CREATED) {
            ERROR_LOG << reply.Code() << "/" << reply.Content() << Endl;
            if (reply.Code() == HTTP_BAD_GATEWAY) {
                return AddTasks<TTaskType>(poolId, tasks, attempt - 1);
            }
            return false;
        }
        return true;
    }

    template <class TTaskType>
    bool GetResults(const TString& poolId, TVector<TTaskType>& tasks, const TString& lastItemId = "") const {
        NNeh::THttpRequest httpRequest = BuildBaseRequest(TGetResultsRequest::GetPath(poolId, lastItemId));
        NUtil::THttpReply reply = ClientSender->SendMessageSync(httpRequest);

        NJson::TJsonValue parsedReply;
        if (reply.Code() != HTTP_OK || !ReadJsonFastTree(reply.Content(), &parsedReply)) {
            ERROR_LOG << reply.Code() << "/" << reply.Content() << Endl;
            return false;
        }

        TString currentItemId;
        for (auto&& item : parsedReply["items"].GetArray()) {
            ui32 lastTaskIndex = tasks.size();
            currentItemId = item["id"].GetString();
            for (auto&& taskData : item["tasks"].GetArray()) {
                TTaskType oneTask;
                Y_ENSURE_BT(oneTask.FromTolokaJson(taskData), "cannot parse " << taskData.GetStringRobust());
                tasks.push_back(oneTask);
            }

            for (auto&& solutionData : item["solutions"].GetArray()) {
                typename TTaskType::TResultType solution;
                Y_ENSURE_BT(solution.FromTolokaJson(solutionData), "cannot parse " << solutionData.GetStringRobust());
                Y_ENSURE_BT(lastTaskIndex < tasks.size(), lastTaskIndex << '/' << tasks.size());
                tasks[lastTaskIndex].SetResults(solution);
                lastTaskIndex++;
            }
        }

        bool hasMore = parsedReply["has_more"].GetBoolean();
        if (hasMore) {
            return GetResults(poolId, tasks, currentItemId);
        }
        return true;
    }

    bool GetPoolInfo(const TString& poolId, bool& isOpened) const;
    bool OpenPool(const TString& poolId) const;
private:
    struct TCreatePoolRequest {
        static TString GetPath() {
            return "api/v1/pools";
        }

        static NJson::TJsonValue GetJsonPost(const TString& projectId, const TString& poolName, const TString& description, TInstant expiredAt, const NJson::TJsonValue& qualityControl) {
            NJson::TJsonValue post;
            post["project_id"] = projectId;
            post["private_name"] = poolName;
            post["private_comment"] = poolName;
            post["public_description"] = description;
            post["may_contain_adult_content"] = false;
            post["will_expire"] = expiredAt.ToStringUpToSeconds();
            post["reward_per_assignment"] = 0.03;
            post["assignment_max_duration_seconds"] = 600;
            post["auto_accept_solutions"] = true;
            post["auto_close_after_complete_delay_seconds"] = 60;
            post["priority"] = 10;
            post["defaults"]["default_overlap_for_new_task_suites"] = 3;
            post["defaults"]["default_overlap_for_new_tasks"] = 3;

            NJson::TJsonValue condition;
            condition["category"] = "profile";
            condition["key"] = "languages";
            condition["operator"] = "IN";
            condition["value"] = "RU";
            NJson::TJsonValue condWrapper;
            condWrapper["or"].AppendValue(condition);
            post["filter"]["and"].AppendValue(condWrapper);

            if (!qualityControl.IsNull()) {
                post["quality_control"] = qualityControl;
            }

            return post;
        }
    };

    struct TCreateSuiteRequest {
        static TString GetPath() {
            return "api/v1/task-suites?open_pool=false";
        }

        template <class TTaskType>
        static NJson::TJsonValue GetJsonPost(const TString& poolId, const TVector<TTaskType>& tasks) {
            NJson::TJsonValue post;
            post["pool_id"] = poolId;
            post["overlap"] = 3;
            post["mixed"] = true;
            post["infinite_overlap"] = false;

            NJson::TJsonValue& tasksJson = post["tasks"];
            for (auto&& task : tasks) {
                tasksJson.AppendValue(task.GetTolokaJson());
            }
            return post;
        }
    };

    struct TOpenPoolRequest {
        static TString GetPath(const TString& poolId) {
            return "api/v1/pools/" + poolId + "/open";
        }

        static NJson::TJsonValue GetJsonPost() {
            NJson::TJsonValue post(NJson::JSON_MAP);
            return post;
        }
    };

    struct TGetResultsRequest {
        static TString GetPath(const TString& poolId, const TString& lastItem) {
            return "api/v1/assignments?sort=id&limit=500&status=ACCEPTED&pool_id=" + poolId + (lastItem ? "&id_gt=" + lastItem : "");
        }

        static NJson::TJsonValue GetJsonPost() {
            NJson::TJsonValue post;
            return post;
        }
    };

    struct TGetPoolInfoRequest {
        static TString GetPath(const TString& poolId) {
            return "api/v1/pools/" + poolId;
        }

        static NJson::TJsonValue GetJsonPost() {
            NJson::TJsonValue post;
            return post;
        }
    };
private:
    const TTolokaClientConfig& Config;
    TSimpleAsyncRequestSender ClientSender;
};
