#include "toloka.h"

bool TFuelingPoolsWrapper::ListTables(const TString& path, bool skipEmpty, TSet<TString>& tables) {
    try {
        NYT::TAttributeFilter attrs;
        attrs.AddAttribute("row_count").AddAttribute("key");
        auto nodes = YtClient->List(path, NYT::TListOptions().AttributeFilter(attrs));
        for (auto&& node : nodes) {
            auto attrs = node.GetAttributes();
            const TString poolName = attrs["key"].AsString();
            ui32 rowCount = attrs["row_count"].AsInt64();
            if (!skipEmpty || rowCount > 0) {
                tables.insert(poolName);
            }
        }
    } catch (const std::exception& e) {
        ERROR_LOG << "YTError " << FormatExc(e) << Endl;
        return false;
    }
    return true;
}

bool TFuelingPoolsWrapper::GetKnownPools(const TSet<TString>& filter, TMap<TString, TString>& pool2Id) {
    try {
        auto reader = YtClient->CreateTableReader<NYT::TNode>(WorkDir + "/pools_info");
        for (; reader->IsValid(); reader->Next()) {
            NYT::TNode inputRow = reader->GetRow();
            TString resultsPoolId = inputRow["pool_id"].AsString();
            TString poolName = inputRow["name"].AsString();
            if (!filter.contains(poolName)) {
                pool2Id[poolName] = resultsPoolId;
            }
        }
    } catch (const std::exception& e) {
        ERROR_LOG << "YTError " << FormatExc(e) << Endl;
        return false;
    }
    return true;
}

bool TFuelingPoolsWrapper::GetPoolResults(const TString& resultsPoolId, const TString& poolName, NDrive::INotifier::TPtr notifier) {
    if (notifier) {
        notifier->Notify(NDrive::INotifier::TMessage("Getting results for pool: " + poolName));
    }

    TVector<TFuelRecognizerTask> results;
    if (!TolokaClient.GetResults(resultsPoolId, results)) {
        if (notifier) {
            notifier->Notify(NDrive::INotifier::TMessage("Get pool results failed for: " + poolName));
        }
        return false;
    }

    TMap<TString, TVector<TFuelRecognizerTask>> resultsGroups;
    for (auto&& res : results) {
        resultsGroups[res.GetTaskId()].push_back(res);
    }

    ui32 rowsCount = 0;
    try {
        auto writer = YtClient->CreateTableWriter<NYT::TNode>(WorkDir + "/toloka_results/" + poolName);
        for (auto&& group : resultsGroups) {
            NYT::TNode node;
            FillResultNode(group.second, node);
            node["work"] = poolName;
            writer->AddRow(node);
            ++rowsCount;
        }
        writer->Finish();
    } catch (const std::exception& e) {
        ERROR_LOG << "YTError " << FormatExc(e) << Endl;
        if (notifier) {
            notifier->Notify(NDrive::INotifier::TMessage("Get pool results failed (YT error) for: " + poolName));
        }
        return false;
    }
    if (notifier) {
        notifier->Notify(NDrive::INotifier::TMessage("Getting results for pool: " + poolName + " finished (with " + ::ToString(rowsCount) + " rows)"));
    }
    return true;
}

bool TFuelingPoolsWrapper::CreatePool(const TGoldenSetGenerator& controlGenerator, const TString& poolName, NDrive::INotifier::TPtr notifier) {
    if (notifier) {
        notifier->Notify(NDrive::INotifier::TMessage("Creating new Toloka pool: " + poolName));
    }

    TMap<ui32, ui32> stat;
    TVector<TFuelRecognizerTask> poolTasks;
    if (!PrepareTasks(WorkDir + "/session/" + poolName, poolTasks, stat)) {
        if (notifier) {
            notifier->Notify(NDrive::INotifier::TMessage("Creating new Toloka pool (prepare tasks): " + poolName));
        }
        return false;
    }

    TString poolId;
    if (!TolokaClient.CreatePool(poolName, "Фотографии заправки", poolId, Config.GetQualityControlConfig())) {
        if (notifier) {
            notifier->Notify(NDrive::INotifier::TMessage("Can't create new Toloka pool: " + poolName));
        }
        return false;
    }

    if (!poolId) {
        if (notifier) {
            notifier->Notify(NDrive::INotifier::TMessage("Can't create new Toloka pool (empty poolId): " + poolName));
        }
        return false;
    }

    TVector<TFuelRecognizerTask> tasks;
    ShuffleRange(poolTasks);
    for (auto&& recognizeTask : poolTasks) {
        tasks.push_back(recognizeTask);

        if (tasks.size() == 3) {
            tasks.push_back(controlGenerator.GetNext());
        }

        if (tasks.size() == 8) {
            tasks.push_back(controlGenerator.GetNext());
        }

        if (tasks.size() == Config.GetTasksInSet()) {
            if (!TolokaClient.AddTasks(poolId, tasks)) {
                return false;
            }
            tasks.clear();
        }
    }

    if (tasks.size() > 0) {
        if (!TolokaClient.AddTasks(poolId, tasks)) {
            if (notifier) {
                notifier->Notify(NDrive::INotifier::TMessage("Can't create new Toloka pool (add tasks): " + poolName));
            }
            return false;
        }
    }

    NJson::TJsonValue meta(NJson::JSON_MAP);
    for (auto&& info : stat) {
        meta["photos_in_session"][ToString(info.first)] = info.second;
    }

    try {
        auto writer = YtClient->CreateTableWriter<NYT::TNode>(NYT::TRichYPath(WorkDir + "/pools_info").Append(true));
        NYT::TNode poolInfo;
        poolInfo["name"] = poolName;
        poolInfo["pool_id"] = poolId;
        poolInfo["timestamp"] = TInstant::Now().Seconds();
        poolInfo["meta"] = NYT::ToNode(meta);
        poolInfo["tasks_count"] = poolTasks.size();
        writer->AddRow(poolInfo);
        writer->Finish();
    } catch (const std::exception& e) {
        ERROR_LOG << "YTError " << FormatExc(e) << Endl;
        if (notifier) {
            meta["pool_id"] = poolId;
            meta["tasks_count"] = poolTasks.size();
            notifier->Notify(NDrive::INotifier::TMessage("Can't save pool stat: " + poolName + " " + meta.GetStringRobust()));
        }
        return 0;
    }

    if (!TolokaClient.OpenPool(poolId)) {
        if (notifier) {
            notifier->Notify(NDrive::INotifier::TMessage("Can't create new Toloka pool (open pool): " + poolName));
        }
        return false;
    }

    if (notifier) {
        notifier->Notify(NDrive::INotifier::TMessage("Pool " + poolName + " successfully created with " + ToString(poolTasks.size()) + " tasks"));
    }
    return true;
}

bool TFuelingPoolsWrapper::CreatePools(const TInstant& maxTs, ui32 maxPoolsCount, NDrive::INotifier::TPtr notifier) {
    const TString resultsPath = WorkDir + "/toloka_results";
    TSet<TString> finishedPools;
    if (!ListTables(resultsPath, true, finishedPools)) {
        if (notifier) {
            notifier->Notify(NDrive::INotifier::TMessage("Can't get list of finished pools from YT"));
        }
        return false;
    }

    TMap<TString, TString> openedPools;
    if (!GetKnownPools(finishedPools, openedPools)) {
        if (notifier) {
            notifier->Notify(NDrive::INotifier::TMessage("Can't get list of known pools from YT"));
        }
        return false;
    }

    ui32 activePoolsCount = 0;

    for (auto&& pool : openedPools) {
        bool isClosedPool = false;
        if (!TolokaClient.GetPoolInfo(pool.second, isClosedPool)) {
            if (notifier) {
                notifier->Notify(NDrive::INotifier::TMessage("Can't read pool info from Toloka: " + pool.second));
            }
            return false;
        }

        if (isClosedPool) {
            GetPoolResults(pool.second, pool.first, notifier);
        } else {
            ++activePoolsCount;
        }
    }

    TString maxSession = TTableSelectorWork::GetSession(maxTs);
    DEBUG_LOG << maxSession << "/" <<  activePoolsCount << Endl;

    const TString sessionsPath = WorkDir + "/session";
    TSet<TString> sessions;
    if (!ListTables(sessionsPath, true, sessions)) {
        if (notifier) {
            notifier->Notify(NDrive::INotifier::TMessage("Can't get list of sessions from YT"));
        }
        return false;
    }

    if (sessions.empty()) {
        return true;
    }

    TMap<TString, TString> pool2Id;
    if (!GetKnownPools(TSet<TString>(), pool2Id)) {
        if (notifier) {
            notifier->Notify(NDrive::INotifier::TMessage("Can't get list of known pools from YT"));
        }
        return false;
    }

    TString trainigPool = *finishedPools.rbegin();
    TGoldenSetGenerator controlGenerator(YtClient, WorkDir + "/toloka_results/" + trainigPool);

    if (controlGenerator.Size() == 0) {
        if (notifier) {
            notifier->Notify(NDrive::INotifier::TMessage("Empty GoldenSet: " + trainigPool));
        }
        return false;
    }

    for (auto&& session : sessions) {
        if (finishedPools.contains(session)) {
            continue;
        }

        DEBUG_LOG << "Current " << session << Endl;

        if (!TTableSelectorWork::Less(session, maxSession)) {
            break;
        }

        if (activePoolsCount == maxPoolsCount) {
            return true;
        }

        if (pool2Id.contains(session)) {
            DEBUG_LOG << "Skip started pool: " << session << Endl;
            continue;
        }

        if (!CreatePool(controlGenerator, session, notifier)) {
            return false;
        }

        ++activePoolsCount;
    }
    return true;
}

 bool TFuelingPoolsWrapper::PrepareTasks(const TString& dataTable, TVector<TFuelRecognizerTask>& results, TMap<ui32, ui32>& stat) {
    try {
        auto reader = YtClient->CreateTableReader<NYT::TNode>(dataTable);
        for (; reader->IsValid(); reader->Next()) {
            NYT::TNode inputRow = reader->GetRow();

            TString userId = inputRow["user_id"].AsString();
            TString carNumber = inputRow["car_number"].AsString();
            TString sessionId = inputRow["session_id"].AsString();
            NYT::TNode report = inputRow["details"];
            stat[report["images"].AsList().size()]++;
            for (auto&& image : report["images"].AsList()) {
                TString imagePath = "https://carsharing-tags.s3.yandex.net/" + image["path"].AsString();
                TFuelRecognizerTask oneTask(carNumber, sessionId, imagePath);
                results.push_back(oneTask);
            }
        }
    } catch (const std::exception& e) {
        ERROR_LOG << "YTError " << FormatExc(e) << Endl;
        return false;
    }
    return true;
}

void TFuelingPoolsWrapper::FillResultNode(const TVector<TFuelRecognizerTask>& results, NYT::TNode& node) {
    TMap<ui32, ui32> values;
    TMap<ui32, ui32> fractions;
    TMap<TString, ui32> types;
    TMap<TString, ui32> numberCheck;

    TString sessionId;
    NJson::TJsonValue fullResultsList;
    NJson::TJsonValue votes;
    for (auto&& task : results) {
        sessionId = task.GetSessionId();
        auto& result = task.GetResults();
        fullResultsList.AppendValue(result.GetTolokaJson());
        values[result.GetValue()]++;
        fractions[result.GetFraction()]++;
        types[result.GetResultType()]++;
        numberCheck[result.GetIsNumberCorrect()]++;
    }

    for (auto&& v : values) {
        if (v.second >= 2) {
            node["value"] = v.first;
            votes["value"] = v.second;
        }
    }

    for (auto&& v : fractions) {
        if (v.second >= 2) {
            node["fraction"] = v.first;
            votes["fraction"] = v.second;
        }
    }

    for (auto&& v : types) {
        if (v.second >= 2) {
            node["type"] = v.first;
            votes["type"] = v.second;
        }
    }

    for (auto&& v : numberCheck) {
        if (v.second >= 2) {
            node["is_number_correct"] = v.first;
            votes["is_number_correct"] = v.second;
        }
    }

    if (!votes.Has("is_number_correct")) {
        votes["is_number_correct"] = 0;
        node["is_number_correct"] = NYT::TNode(NYT::TNode::CreateEntity());
    }
    if (!votes.Has("type")) {
        votes["type"] = 0;
        node["type"] = NYT::TNode(NYT::TNode::CreateEntity());
    }
    if (!votes.Has("value")) {
        votes["value"] = 0;
        node["value"] = NYT::TNode(NYT::TNode::CreateEntity());
    }
    if (!votes.Has("fraction")) {
        votes["fraction"] = 0;
        node["fraction"] = NYT::TNode(NYT::TNode::CreateEntity());
    }

    node["session_id"] = results.front().GetSessionId();
    node["image_url"] = results.front().GetImageUrl();
    node["car_number"] = results.front().GetCarNumber();
    node["raw_replies"] = NYT::ToNode(fullResultsList);;
    node["votes"] = NYT::ToNode(votes);;
}
