#pragma once

#include <drive/library/cpp/toloka/client.h>

#include <drive/library/cpp/yt/node/cast.h>

#include <drive/backend/abstract/notifier.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/logger/global/global.h>

#include <mapreduce/lib/init.h>
#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/interface/init.h>
#include <mapreduce/yt/interface/io-inl.h>
#include <mapreduce/yt/interface/node.h>

#include <util/generic/set.h>
#include <util/random/shuffle.h>
#include <util/stream/file.h>

struct TTableSelectorWork {
    public:
        static TString GetTable(const TInstant& ts) {
            TInstant dayStart = TInstant::Days(ts.Days());
            ui32 hour = ts.Hours() - dayStart.Hours();
            if (hour >= 6 && hour < 18) {
                return TString(dayStart.ToString().data(), 10) + "_morning";
            }

            if (hour < 6) {
                dayStart = dayStart - TDuration::Days(1);
            }
            return TString(dayStart.ToString().data(), 10) + "_night";
        }

        static TString GetSession(const TInstant& ts) {
            return GetTable(ts);
        }

        static bool Less(const TString& session1, const TString& session2) {
            TInstant inst1 = TInstant::ParseIso8601(TStringBuf(session1.data(), 10));
            TInstant inst2 = TInstant::ParseIso8601(TStringBuf(session2.data(), 10));
            if (inst1 == inst2) {
                if (session1.EndsWith("morning") && session2.EndsWith("night")) {
                    return true;
                }
            }
            return inst1 < inst2;
        }
};

class TRecognizerResults {
    R_FIELD(ui32, Value, Max<ui32>());
    R_FIELD(ui32, Fraction, Max<ui32>());
    R_FIELD(TString, ResultType, "unknown");
    R_FIELD(TString, IsNumberCorrect, "unknown");

public:
    TRecognizerResults() = default;

    bool FromTolokaJson(const NJson::TJsonValue& taskData) {
        const NJson::TJsonValue& data = taskData["output_values"];
        ResultType = data["result_type"].GetString();
        IsNumberCorrect = data["is_number_correct"].GetString();
        const NJson::TJsonValue& resultFrac = data["result_frac"];
        const NJson::TJsonValue& resultValue = data["result_value"];
        if (resultFrac.IsDefined() && !TryFromString(resultFrac.GetStringRobust(), Fraction)) {
            return false;
        }
        if (resultValue.IsDefined() && !TryFromString(resultValue.GetStringRobust(), Value)) {
            return false;
        }
        return true;
    }

    NJson::TJsonValue GetTolokaJson() const {
        NJson::TJsonValue taskData;
        taskData["result_type"] = ResultType;
        taskData["is_number_correct"] = IsNumberCorrect;
        if (Value != Max<ui32>()) {
            taskData["result_value"] = Value;
        }
        if (Fraction != Max<ui32>()) {
            taskData["result_frac"] = Fraction;
        }

        NJson::TJsonValue result;
        result["output_values"] = taskData;
        return result;
    }
};

class TFuelRecognizerTask {
    R_READONLY(TString, CarNumber);
    R_READONLY(TString, TaskId);
    R_READONLY(TString, SessionId);
    R_READONLY(TString, ImageUrl);
    R_READONLY(TInstant, Timestamp, TInstant::Zero());
    TMaybe<TRecognizerResults> Results;

public:
    using TResultType = TRecognizerResults;

    TFuelRecognizerTask(const TString& carNumber, const TString& sessionId, const TString& imageUrl)
        : CarNumber(carNumber)
        , SessionId(sessionId)
        , ImageUrl(imageUrl) {}

    TFuelRecognizerTask() = default;

    void SetResults(const TRecognizerResults& results) {
        Results = results;
    }

    const TRecognizerResults& GetResults() const {
        return Results.GetRef();
    }

    bool FromTolokaJson(const NJson::TJsonValue& taskData) {
        TaskId = taskData["id"].GetString();
        CarNumber = taskData["input_values"]["car_number"].GetString();
        SessionId = taskData["input_values"]["session_id"].GetString();
        ImageUrl = taskData["input_values"]["image_url"].GetString();
        return true;
    }

    NJson::TJsonValue GetTolokaJson() const {
        NJson::TJsonValue taskData;
        taskData["car_number"] = CarNumber;
        taskData["session_id"] = SessionId;
        taskData["image_url"] = ImageUrl;

        NJson::TJsonValue result;
        result["input_values"] = taskData;
        if (Results.Defined()) {
            result["known_solutions"].AppendValue(Results->GetTolokaJson());
        }
        return result;
    }
};

class TGoldenSetGenerator {
public:
    TGoldenSetGenerator(NYT::IClientPtr ytClient, const TString& tableWithExamples) {
        auto reader = ytClient->CreateTableReader<NYT::TNode>(tableWithExamples);

        for (; reader->IsValid(); reader->Next()) {
            NYT::TNode inputRow = reader->GetRow();
            TString sessionId = inputRow["session_id"].AsString();
            TString carNumber = inputRow["car_number"].AsString();
            TString imageUrl = inputRow["image_url"].AsString();
            if (!inputRow["type"].IsString() || !inputRow["is_number_correct"].IsString())
                continue;

            TString type = inputRow["type"].AsString();
            TString isNumberCorrect = inputRow["is_number_correct"].AsString();

            NJson::TJsonValue votes = NYT::FromNode<NJson::TJsonValue>(inputRow["votes"]);
            if (votes.GetMap().size() != 4) {
                ERROR_LOG << "Skip session: " << sessionId << Endl;
                continue;
            }

            bool isFullConfirmed = true;
            for (auto&& vote : votes.GetMap()) {
                if (vote.second != 3) {
                    isFullConfirmed = false;
                }
            }

            if (!isFullConfirmed) {
                continue;
            }

            NJson::TJsonValue rawResults = NYT::FromNode<NJson::TJsonValue>(inputRow["raw_replies"]);
            CHECK_WITH_LOG(rawResults.GetArray().size() == 3) << NYT::FromNode<NJson::TJsonValue>(inputRow).GetStringRobust();
            TFuelRecognizerTask oneTask(carNumber, sessionId, imageUrl);
            TRecognizerResults results;
            results.FromTolokaJson(rawResults[0]);
            oneTask.SetResults(results);
            Tasks.push_back(oneTask);
        }
        CHECK_WITH_LOG(Tasks.size() > 0);
    }

    ui32 Size() const {
        return Tasks.size();
    }

    const TFuelRecognizerTask& GetNext() const {
        ++Index;
        if (Index == Tasks.size()) {
            Index = 0;
            ShuffleRange(Tasks);
        }
        return Tasks[Index];
    }

private:
    mutable TVector<TFuelRecognizerTask> Tasks;
    mutable ui32 Index = 0;
};


class TFuelingPoolsWrapper {
public:
    TFuelingPoolsWrapper(const TTolokaClientConfig& config, NYT::IClientPtr ytClient, const TString& workDir = "//home/extdata/nsofya/fueling")
        : Config(config)
        , TolokaClient(config)
        , YtClient(ytClient)
        , WorkDir(workDir)
    {
    }

    bool ListTables(const TString& path, bool skipEmpty, TSet<TString>& result);

    bool GetKnownPools(const TSet<TString>& filteri, TMap<TString, TString>& result);

    bool GetPoolResults(const TString& resultsPoolId, const TString& poolName, TAtomicSharedPtr<NDrive::INotifier> notifier = nullptr);

    bool CreatePool(const TGoldenSetGenerator& controlGenerator, const TString& poolName, TAtomicSharedPtr<NDrive::INotifier> notifier = nullptr);

    bool CreatePools(const TInstant& maxTs, ui32 maxPoolsCount = 1, TAtomicSharedPtr<NDrive::INotifier> notifier = nullptr);

private:
    bool PrepareTasks(const TString& dataTable, TVector<TFuelRecognizerTask>& results, TMap<ui32, ui32>& stat);

    void FillResultNode(const TVector<TFuelRecognizerTask>& results, NYT::TNode& node);

private:
    TTolokaClientConfig Config;
    TTolokaClient TolokaClient;
    NYT::IClientPtr YtClient;
    TString WorkDir;
};
