#include "action.h"

#include <drive/library/cpp/common/status.h>

TAtomicSharedPtr<NRTLine::TAction> NDrive::CreateAction(ui64 imei, const NYT::TNode& data, const NDrive::TStatus& status) {
    auto timestamp = NDrive::GetTimestamp(data);
    auto action = MakeAtomicShared<NRTLine::TAction>();
    NRTLine::TDocument& document = action->AddDocument();
    NRTLine::TGeoData& geodata = document.AddGeoData();

    document.AddProperty("imei", imei);
    document.AddProperty("destination", TGeoCoord().ToString());
    if (auto g_x = NDrive::GetCustomParameter(data, "1243")) {
        document.AddProperty("g_x", g_x);
    }
    if (auto g_y = NDrive::GetCustomParameter(data, "1244")) {
        document.AddProperty("g_y", g_y);
    }
    if (auto g_z = NDrive::GetCustomParameter(data, "1245")) {
        document.AddProperty("g_z", g_z);
    }

    Y_ENSURE(status.GetActionId(), imei << '\t' << status);
    TString url;
    if (status.GetActionPrefix()) {
        url.append(status.GetActionPrefix());
        url.append('-');
    } else if (status.GetCarStatus() != "ride" && status.GetCarStatus() != "riding") {
        url.append(status.GetCarStatus());
        url.append('-');
    }
    url.append(status.GetActionId());
    if (status.GetActionSuffix()) {
        url.append('-');
        url.append(status.GetActionSuffix());
    }
    document.AddProperty("RHASH", url);
    document.SetUrl(url);

    document.AddProperty("Status", NDrive::GetStatus(status.GetCarStatus()));

    if (status.GetCarId()) {
        document.AddProperty("DID", status.GetCarId());
        document.AddProperty("s_device_id", status.GetCarId());
    }
    if (status.GetSessionId()) {
        document.AddProperty("SESSIONID", status.GetSessionId());
    }
    if (status.GetUserId()) {
        document.AddProperty("ACCID", status.GetUserId());
    }

    double speed = NDrive::GetSpeed(data);
    if (auto canSpeed = NDrive::GetCustomParameter(data, "2111")) {
        document.AddProperty("s_can_speed", canSpeed);
        document.AddProperty("s_raw_speed", speed);
        if (*canSpeed > 0) {
            speed = std::min(speed, *canSpeed);
        }
    }
    document.AddProperty("s_speed", speed);

    document.AddProperty("TS", 0.001 * timestamp.MilliSeconds());
    document.SetTimestamp(timestamp.Seconds());
    document.AddProperty("ModificationType", "mtTracePoint");

    geodata.AddCoordinate({
        NDrive::GetLongitude(data),
        NDrive::GetLatitude(data)
    });
    geodata.SetType("DRIVE_RIDING");

    return action;
}

TAtomicSharedPtr<NRTLine::TAction> NDrive::CreateAction(ui64 imei, const NYT::TNode& data, TAtomicSharedPtr<NDrive::TBaseTimeline> timeline, const TString& suffix, bool skipAvailable) {
    auto timestamp = NDrive::GetTimestamp(data);
    auto status = timeline->GetStatus(timestamp);
    if (status.GetCarStatus() == "available") {
        if (skipAvailable) {
            return nullptr;
        }
        if (status.GetStart() > TInstant::Zero()) {
            DEBUG_LOG << "IMEI " << imei << ": requesting previous status for " << status << Endl;
            status = timeline->GetPreviousStatus(status);
            DEBUG_LOG << "IMEI " << imei << ": previous status is " << status << Endl;
            status.SetCarStatus("available");
            status.SetActionPrefix("post");
        } else {
            DEBUG_LOG << "IMEI " << imei << ": set ActionId" << Endl;
            status.SetActionId(status.GetCarId());
            status.SetUserId("null");
        }
    }
    if (status.GetCarStatus() == "incorrect") {
        WARNING_LOG << "IMEI " << imei << ": skip incorrect status" << Endl;
        return nullptr;
    }
    if (suffix) {
        status.SetActionSuffix(suffix);
    }
    return CreateAction(imei, data, status);
}
