#include "client.h"

#include "tio.h"

#include <drive/library/cpp/maps_router/linker.h>
#include <drive/library/cpp/threading/future.h>

#include <rtline/library/json/cast.h>

NDrive::TTracksClient::TTracksClient(const TString& service, const TString& host, ui16 port)
    : LocalSearchClient(MakeHolder<NRTLine::TNehSearchClient>(service, host, port))
    , SearchClient(*LocalSearchClient)
{
}

NDrive::TTracksClient::TTracksClient(const NRTLine::TNehSearchClient& client)
    : SearchClient(client)
{
}

NThreading::TFuture<NDrive::TTracks> NDrive::TTracksClient::GetTracks(const NDrive::TTrackQuery& query, TDuration timeout) const {
    auto trackQuery = NDrive::MakeTrackQuery(query);
    trackQuery.SetReqClass(ReqClass);
    trackQuery.SetTimeout(timeout);
    auto reply = SearchClient.SendAsyncQueryF(trackQuery, timeout);
    auto source = SearchClient.GetServiceName();
    auto result = reply.Apply([query, source = std::move(source)](const NThreading::TFuture<NRTLine::TSearchReply>& r) {
        const auto& reply = r.GetValue();
        Y_ENSURE(reply.IsSucceeded(), reply.GetCode() << ' ' << reply.GetReqId());
        NDrive::TTrackVisitor visitor;
        if (query.RideId) {
            visitor.SetRideId(query.RideId);
        }
        if (query.Status) {
            visitor.SetStatus(*query.Status);
        }
        if (query.Since != TInstant::Zero()) {
            visitor.SetSince(query.Since);
        }
        if (query.Until != TInstant::Max()) {
            visitor.SetUntil(query.Until);
        }
        TReportAccessor scanner(reply.GetReport());
        scanner.Visit(visitor);
        auto tracks = visitor.ExtractTracks();
        for (auto&& track : tracks) {
            if (CalcDuration(track) < query.DurationThreshold) {
                continue;
            }
            track.Source = source;
        }
        return tracks;
    });
    return result;
}

namespace {
    THolder<NGraph::ILinker> CreateLinker(const NDrive::TTracksLinker::TConfig& config) {
        auto legacy = config.Host ? MakeHolder<NGraph::TRouter>(config.Host, config.Port, config.Service) : nullptr;
        auto maps = config.MapsLinkerEndpoint ? MakeHolder<NDrive::TMapsLinker>(config.MapsLinkerEndpoint, std::move(legacy)) : nullptr;
        if (maps) {
            return maps;
        } else {
            return legacy;
        }
    }
}

NDrive::TTracksLinker::TTracksLinker(const TConfig& config, const TOptions& options)
    : Linker(CreateLinker(config))
    , Options(options)
{
    Y_ENSURE(Linker, "cannot CreateLinker");
}

NDrive::TTracksLinker::TTracksLinker(TAtomicSharedPtr<NGraph::ILinker> linker, const TOptions& options)
    : Linker(std::move(linker))
    , Options(options)
{
}

NDrive::TTracksLinker::~TTracksLinker() {
}

NThreading::TFuture<NDrive::TTracksLinker::TResults> NDrive::TTracksLinker::Link(NThreading::TFuture<TTracks>&& tracks) const {
    return tracks.Apply([tracks, linker = Linker, options = Options](const NThreading::TFuture<TTracks>& /*t*/) mutable {
        Y_ENSURE(linker);
        return Link(tracks.ExtractValue(), *linker, options);
    });
}

NThreading::TFuture<NDrive::TTracksLinker::TResults> NDrive::TTracksLinker::Link(TTracks&& tracks) const {
    return Link(std::move(tracks), *Linker, Options);
}

NThreading::TFuture<NDrive::TTracksLinker::TResults> NDrive::TTracksLinker::Link(TTracks&& tracks, const NGraph::ILinker& linker, const TOptions& options) {
    NThreading::TFutures<TResult> results;
    for (auto&& track : tracks) {
        results.push_back(Link(std::move(track), linker, options));
    }
    return NThreading::Merge(std::move(results));
}

NThreading::TFuture<NDrive::TTracksLinker::TResult> NDrive::TTracksLinker::Link(NThreading::TFuture<TTrack>&& track) const {
    return track.Apply([track, linker = Linker, options = Options](const NThreading::TFuture<TTrack>& /*t*/) mutable {
        Y_ENSURE(linker);
        return Link(track.ExtractValue(), *linker, options);
    });
}

NThreading::TFuture<NDrive::TTracksLinker::TResult> NDrive::TTracksLinker::Link(TTrack&& track) const {
    return Link(std::move(track), *Linker, Options);
}

NThreading::TFuture<NDrive::TTracksLinker::TResult> NDrive::TTracksLinker::Link(TTrack&& track, const NGraph::ILinker& linker, const TOptions& options) {
    auto filtered = FilterCoordinates(track.Coordinates, options.FilterSpan, options.FilterThreshold);
    auto segments = SegmentCoordinates(filtered, options.SegmentSize, options.SplitThreshold);

    TResult result;
    result.Track = std::move(track);
    result.FilteredCoordinates = std::move(filtered);
    for (auto&& coordinates : segments) {
        result.Segments.emplace_back().Coordinates = std::move(coordinates);
    }

    NThreading::TFutures<NGraph::TRouter::TMatch> matches;
    NGraph::TRouter::TMatchingOptions matchingOptions;
    matchingOptions.Debug = false;
    matchingOptions.DistancePrecision = options.Precision;
    for (auto&& segment : result.Segments) {
        matches.push_back(linker.Match(segment.Coordinates, matchingOptions));
    }
    NThreading::TFuture<void> waiter = NThreading::WaitExceptionOrAll(matches);

    return waiter.Apply([options, matches = std::move(matches), result = std::move(result)](const NThreading::TFuture<void>& /*w*/) mutable {
        Y_ENSURE(matches.size() == result.Segments.size());
        for (size_t i = 0; i < result.Segments.size(); ++i) {
            TSegment& segment = result.Segments[i];
            segment.Linked = matches[i].ExtractValue();

            TSpeedLimitRanges ranges = AnalyzeSpeedLimit(segment.Linked, options.ViolationOptions);
            for (auto&& range : ranges) {
                TSpeedLimitRanges split = SplitViolations(range, segment.Coordinates, options.ViolationOptions);
                for (auto&& s : split) {
                    segment.Processed.push_back(std::move(s));
                }
            }
        }
        return result;
    });
}

NThreading::TFuture<NDrive::TTracks> NDrive::TMetaTracksClient::GetTracks(const NDrive::TTrackQuery& query, TDuration timeout) const {
    NThreading::TFutures<NDrive::TTracks> tracks;
    for (auto&& client : Clients) {
        tracks.push_back(Yensured(client)->GetTracks(query, timeout));
    }
    auto waiter = NThreading::WaitAll(tracks);
    return waiter.Apply([tracks = std::move(tracks)](const NThreading::TFuture<void>& w) {
        for (auto&& track : tracks) {
            if (track.HasValue() && !track.GetValue().empty()) {
                return track;
            }
        }
        for (auto&& track : tracks) {
            if (track.HasValue()) {
                return track;
            }
        }
        for (auto&& track : tracks) {
            return track;
        }
        w.GetValue();
        throw yexception() << "no async tracks";
    });
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TTracksLinker::TResult& object) {
    NJson::TJsonValue result = NJson::JSON_MAP;
    result["filtered_coordinates_size"] = object.FilteredCoordinates.size();
    result["segments_size"] = object.Segments.size();
    return result;
}

TMaybe<TRange<TInstant>> NDrive::TTrackOps::GetLinkedTrackRange(const TLinkedTrack& track) {
    if (track.FilteredCoordinates.empty()) {
        return {};
    }
    TRange<TInstant> range(TInstant::Max(), TInstant::Zero());
    for (const auto& coord : track.FilteredCoordinates) {
        if (coord.Timestamp > range.To.GetRef()) {
            range.To = coord.Timestamp;
        }
        if (coord.Timestamp < range.From.GetRef()) {
            range.From = coord.Timestamp;
        }
    }
    return range;
}

TMaybe<TRange<TInstant>> NDrive::TTrackOps::GetLinkedTracksRange(const TLinkedTracks& tracks) {
    TRange<TInstant> range(TInstant::Max(), TInstant::Zero());
    bool rangeSet = false;
    for (const auto& track : tracks) {
        if (auto trackRange = GetLinkedTrackRange(track); trackRange) {
            rangeSet = true;
            if (trackRange->From.GetRef() < range.From.GetRef()) {
                range.From = trackRange->From;
            }
            if (trackRange->To.GetRef() > range.To.GetRef()) {
                range.To = trackRange->To;
            }
        }
    }
    if (!rangeSet) {
        return {};
    }
    return range;
}

bool NDrive::TTrackOps::IsOverlapping(const TTrack& track, const TRange<TInstant>& range) {
    if (track.Coordinates.empty() || !range.From || !range.To) {
        return false;
    }

    return !(
        (track.Coordinates.front().Timestamp >= (*range.To)) ||
        (track.Coordinates.back().Timestamp <= (*range.From))
    );
}

TVector<TRange<TInstant>> NDrive::TTrackOps::BuildLinkedTracksRanges(const TLinkedTracks& tracks) {
    TMap<TInstant, int> rangesMarks;
    for (const TLinkedTrack& track : tracks) {
        if (track.FilteredCoordinates.empty()) {
            continue;
        }

        if (auto trackRange = GetLinkedTrackRange(track)) {
            ++rangesMarks[trackRange->From.GetRef()];
            --rangesMarks[trackRange->To.GetRef()];
        }
    }

    TVector<TRange<TInstant>> ranges;
    TMaybe<TRange<TInstant>> targetRange;
    int rangeCounter = 0;
    for (auto&& mark : rangesMarks) {
        Y_ENSURE(rangeCounter >= 0);
        rangeCounter += mark.second;
        if (!targetRange) {
            targetRange.ConstructInPlace(mark.first);
        }
        if (rangeCounter == 0) {
            targetRange->To = mark.first;
            ranges.push_back(std::move(*targetRange.Get()));
            targetRange.Clear();
        }
    }

    return ranges;
}

NDrive::TTracks NDrive::TTrackOps::SubtractRanges(const TTrack& track, const TRange<TInstant>& range) {
    if (!range.From || !range.To) {
        return {track};
    }

    const auto& until = *range.From;
    const auto& since = *range.To;

    NDrive::TTrack headTrack = CropTrack(track, TInstant::Zero(), until);
    NDrive::TTrack tailTrack = CropTrack(track, since, TInstant::Max());

    NDrive::TTracks tracks;
    if (!headTrack.Coordinates.empty()) {
        tracks.push_back(std::move(headTrack));
    }
    if (!tailTrack.Coordinates.empty()) {
        tracks.push_back(std::move(tailTrack));
    }
    return tracks;
}

NDrive::TTracks NDrive::TTrackOps::SubtractLinked(const TTracks& tracks, const TLinkedTracks& linkedTracks) {
    const auto ranges = BuildLinkedTracksRanges(linkedTracks);

    NDrive::TTracks complementaryTracks;
    for (const auto& trackToSubtract : tracks) {
        complementaryTracks.push_back(trackToSubtract);

        bool isOverlapped = false;
        do {
            isOverlapped = false;
            const auto& track = complementaryTracks.back();
            for (const auto& range : ranges) {
                if (IsOverlapping(track, range)) {
                    auto subtractedTracks = SubtractRanges(track, range);
                    complementaryTracks.pop_back();
                    complementaryTracks.insert(
                        complementaryTracks.end(),
                        std::make_move_iterator(subtractedTracks.begin()),
                        std::make_move_iterator(subtractedTracks.end()));
                    isOverlapped = true;
                    break;
                }
            }
        } while (isOverlapped);
    }
    return complementaryTracks;
}

void NDrive::TTrackOps::SortLinkedTracks(TLinkedTracks& linkedTracks) {
    auto comp = [](
        const TLinkedTrack& first,
        const TLinkedTrack& second
    ) {
        if (second.FilteredCoordinates.empty()) {
            return first.FilteredCoordinates.empty();
        } else if (first.FilteredCoordinates.empty()) {
            return true;
        }

        return
            first.FilteredCoordinates.begin()->Timestamp <
            second.FilteredCoordinates.begin()->Timestamp;
    };
    std::sort(linkedTracks.begin(), linkedTracks.end(), comp);
}
