#pragma once

#include "interface.h"

#include <rtline/util/types/range.h>

namespace NDrive {
    class TTracksClient: public ITrackClient {
    public:
        TTracksClient(const TString& service, const TString& host, ui16 port);
        TTracksClient(const NRTLine::TNehSearchClient& client);

        NThreading::TFuture<TTracks> GetTracks(const TTrackQuery& query, TDuration timeout) const override;

    private:
        THolder<NRTLine::TNehSearchClient> LocalSearchClient;
        const NRTLine::TNehSearchClient& SearchClient;

        TString ReqClass = "TRACKCL";
    };

    class TTracksLinker {
    public:
        struct TSegment {
            NGraph::TRouter::TTimedGeoCoordinates Coordinates;
            NGraph::TRouter::TMatch Linked;
            TSpeedLimitRanges Processed;
        };
        struct TResult {
            TTrack Track;
            NGraph::TRouter::TTimedGeoCoordinates FilteredCoordinates;
            TVector<TSegment> Segments;
        };
        using TResults = TVector<TResult>;

        struct TConfig {
            TString MapsLinkerEndpoint = "http://core-jams-matcher.testing.maps.yandex.net";
            TString Host = "saas-searchproxy-maps-prestable.yandex.net";
            TString Service = "drive_router";
            ui16 Port = 17000;
        };

        struct TOptions {
            TAnalyzeSpeedLimitOptions ViolationOptions;
            double FilterThreshold = 10000;
            double SplitThreshold = 5000;
            ui32 FilterSpan = 500;
            ui32 Precision = 20;
            ui32 SegmentSize = 300;
        };

    public:
        TTracksLinker(const TConfig& config, const TOptions& options = Default<TOptions>());
        TTracksLinker(TAtomicSharedPtr<NGraph::ILinker> linker, const TOptions& options = Default<TOptions>());
        ~TTracksLinker();

        NThreading::TFuture<TResults> Link(NThreading::TFuture<TTracks>&& tracks) const;
        NThreading::TFuture<TResults> Link(TTracks&& tracks) const;
        NThreading::TFuture<TResult> Link(NThreading::TFuture<TTrack>&& track) const;
        NThreading::TFuture<TResult> Link(TTrack&& track) const;

    private:
        static NThreading::TFuture<TResults> Link(TTracks&& tracks, const NGraph::ILinker& linker, const TOptions& options);
        static NThreading::TFuture<TResult> Link(TTrack&& track, const NGraph::ILinker& linker, const TOptions& options);

    private:
        TAtomicSharedPtr<NGraph::ILinker> Linker;
        TOptions Options;
    };

    class TTrackOps {
    public:
        using TSegment = TTracksLinker::TSegment;
        using TLinkedTrack = TTracksLinker::TResult;
        using TLinkedTracks = TTracksLinker::TResults;

    public:
        static TMaybe<TRange<TInstant>> GetLinkedTrackRange(const TLinkedTrack& track);
        static TMaybe<TRange<TInstant>> GetLinkedTracksRange(const TLinkedTracks& tracks);

        static bool IsOverlapping(const TTrack& track, const TRange<TInstant>& range);
        static TVector<TRange<TInstant>> BuildLinkedTracksRanges(const TLinkedTracks& tracks);
        static TTracks SubtractRanges(const TTrack& track, const TRange<TInstant>& range);
        static TTracks SubtractLinked(const TTracks& tracks, const TLinkedTracks& linkedTracks);
        static void SortLinkedTracks(TLinkedTracks& linkedTracks);
    };

    class TMetaTracksClient: public ITrackClient {
    public:
        using TClients = TVector<TTrackClientPtr>;

    public:
        TMetaTracksClient(TClients&& clients)
            : Clients(std::move(clients))
        {
        }

        NThreading::TFuture<TTracks> GetTracks(const TTrackQuery& query, TDuration timeout) const override;

    private:
        TClients Clients;
    };
}
