#include "projector.h"

namespace {

    static const int POLYLINE_RADIUS_POINTS_LIMIT = 100;

}

namespace NDrive {

    TTrackProjector::TTrackProjector(const NDrive::TTracksLinker::TResults& results)
    {
        for (auto&& result : results) {
            for (auto&& segment : result.Segments) {
                if (segment.Coordinates.empty()) {
                    continue;
                }
                size_t it = 0;
                TVector<NGraph::TTimedGeoCoordinate> points;
                for (auto&& range : segment.Processed) {
                    for (size_t i = 0; i < range.Points.size(); ++i) {
                        const auto& point = range.Points[i];
                        // Approximate nearest raw point for match.
                        while (it + 1 < segment.Coordinates.size() && point.GetLengthTo(segment.Coordinates[it + 1]) < point.GetLengthTo(segment.Coordinates[it])) {
                            ++it;
                        }
                        points.emplace_back(point, segment.Coordinates[it].Timestamp);
                    }
                }
                if (!points.empty()) {
                    Coordinates.insert(Coordinates.end(), points.begin(), points.end());
                    continue;
                }
                for (auto&& coordinate : segment.Coordinates) {
                    Coordinates.push_back(coordinate);
                }
            }
        }
        std::stable_sort(Coordinates.begin(), Coordinates.end(), [](const NGraph::TTimedGeoCoordinate& lhs, const NGraph::TTimedGeoCoordinate& rhs) {
            return lhs.Timestamp < rhs.Timestamp;
        });
    }

    TGeoCoord TTrackProjector::Project(const TGeoCoord& coord, TInstant ts, TDuration delta /*= TDuration::Seconds(30)*/) const {
        auto it = std::lower_bound(
            Coordinates.begin(), Coordinates.end(), ts,
            [](const NGraph::TTimedGeoCoordinate& x, TInstant v) {
                return x.Timestamp < v;
            }
        );
        auto lo = it;
        for (int i = 0; i < POLYLINE_RADIUS_POINTS_LIMIT && lo != Coordinates.begin(); ++i) {
            --lo;
            if (ts - lo->Timestamp > delta) {
                break;
            }
        }
        auto hi = it;
        for (int i = 0; i < POLYLINE_RADIUS_POINTS_LIMIT && hi != Coordinates.end(); ++i) {
            if (hi->Timestamp - ts > delta) {
                break;
            }
            ++hi;
        }
        if (lo == hi) {
            return coord;
        }
        TPolyLine<TGeoCoord> polyline(TVector<TGeoCoord>(lo, hi));
        TPolyLine<TGeoCoord>::TProjectInfo info;
        if (!polyline.Project(coord, info)) {
            return coord;
        }
        return info.Coord;
    }

}
