#include "tio.h"
#include "trace.h"

#include <library/cpp/string_utils/parse_vector/vector_parser.h>
#include <library/cpp/protobuf/json/proto2json.h>

namespace {
    const TString DeviceIdProperty = "s_device_id";
    const TString UserIdProperty = "s_user_id";

    const TString CoordsProp = "TrackCoords";
    const TString TimestampsProp = "TimestampsCoords";
    const TString SpeedsProp = "SpeedsCoords";
}

TReportAccessor::EStatus NDrive::TTrackVisitor::VisitDoc(const TGrouping& /*grouping*/, const TGroup& /*group*/, const TDocument& doc) {
    TTrack track;
    TVector<TGeoCoord> coordinates;
    TVector<ui64> timestamps;
    TVector<float> speeds;
    track.Name = doc.GetArchiveInfo().GetUrl();
    track.Status = NDrive::GetStatus(track.Name);
    if (Status && track.Status != Status) {
        return TReportAccessor::EStatus::sNext;
    }
    if (RideId && track.Name != RideId) {
        return TReportAccessor::EStatus::sNext;
    }
    for (auto&& propertie : doc.GetArchiveInfo().GetGtaRelatedAttribute()) {
        const TString& key = propertie.GetKey();
        const TString& value = propertie.GetValue();
        if (key == CoordsProp) {
            Y_ENSURE(TGeoCoord::DeserializeVector(value, coordinates));
            continue;
        }
        if (key == TimestampsProp) {
            Y_ENSURE(TryParseStringToVector(value, timestamps, ' '));
            continue;
        }
        if (key == SpeedsProp) {
            Y_ENSURE(TryParseStringToVector(value, speeds, ' '));
            for (auto&& speed : speeds) {
                speed /= 3.6;
            }
            continue;
        }
        if (key == UserIdProperty) {
            track.UserId = value;
            continue;
        }
        if (key == DeviceIdProperty) {
            track.DeviceId = value;
            continue;
        }
        if (key == "s_session_id") {
            track.SessionId = value;
            continue;
        }
        if (key == "Trace") {
            size_t index = 0;
            auto parser = [&](const NDrive::TTracePoint& trace) {
                coordinates.push_back(trace.Coordinate);
                timestamps.push_back(trace.Timestamp.Seconds());
                speeds.push_back(trace.Speed / 3.6);
                if (trace.Timestamp < Since || trace.Timestamp > Until) {
                    return;
                }

                ++index;
                if (!trace.ExternalId) {
                    return;
                }

                if (!track.Linked) {
                    track.Linked.ConstructInPlace();
                }
                auto& match = *track.Linked;
                if (match.Elements.empty() || match.Elements[0].EdgeExternalId != trace.ExternalId) {
                    match.Elements.emplace_back();
                    match.Elements.back().EdgeExternalId = trace.ExternalId;
                }

                auto& edge = match.Elements.back();
                auto flow = std::max(trace.FlowSpeed, 0.1);
                auto limit = std::min(trace.SpeedLimit, 100.0);
                edge.Length = trace.EdgeLength;
                edge.Features[NGraph::FI_LENGTH] = trace.EdgeLength;
                edge.Features[NGraph::FI_DURATION] = trace.EdgeLength / flow;
                edge.Features[NGraph::FI_SPEED_LIMIT_DURATION] = trace.EdgeLength / limit;
                edge.Features[NGraph::FI_TRACKS_COUNT] = trace.FlowTracksCount;
                edge.Points.push_back(trace.Coordinate);
                edge.RawIndices.push_back(index);
            };
            Y_ENSURE(ParseTrace(value, parser));
            continue;
        }
    }
    Y_ENSURE(coordinates.size() == timestamps.size());
    Y_ENSURE(coordinates.size() == speeds.size());
    track.Coordinates.reserve(coordinates.size());
    for (size_t i = 0; i < coordinates.size(); ++i) {
        auto timestamp = TInstant::Seconds(timestamps[i]);
        if (timestamp < Since || timestamp > Until) {
            continue;
        }
        track.Coordinates.emplace_back(
            coordinates[i],
            timestamp,
            speeds[i]
        );
    }
    if (!track.Coordinates.empty()) {
        Tracks.push_back(std::move(track));
    }
    return TReportAccessor::EStatus::sNext;
}

NRTLine::TQuery NDrive::MakeTrackQuery(const TString& deviceId, const TString& rideId, const TString& userId, const TString& sessionId, TInstant since, TInstant until, ui32 numdoc) {
    TTrackQuery trackQuery;
    trackQuery.DeviceId = deviceId;
    trackQuery.RideId = rideId;
    trackQuery.UserId = userId;
    trackQuery.SessionId = sessionId;
    trackQuery.Since = since;
    trackQuery.Until = until;
    trackQuery.NumDoc = numdoc;
    return MakeTrackQuery(trackQuery);
}

NRTLine::TQuery NDrive::MakeTrackQuery(const TTrackQuery& trackQuery) {
    const TString& deviceId = trackQuery.DeviceId;
    const TString& rideId = trackQuery.RideId;
    const TString& sessionId = trackQuery.SessionId;
    const TString& userId = trackQuery.UserId;
    const TInstant since = trackQuery.Since;
    const TInstant until = trackQuery.Until;
    const auto numdoc = trackQuery.NumDoc;

    TString text;
    if (deviceId) {
        text += Sprintf("s_device_id:\"%s\" ", deviceId.c_str());
    }
    if (rideId) {
        text += Sprintf("url:\"%s\" ", rideId.c_str());
    }
    if (sessionId) {
        text += Sprintf("s_session_id:\"%s\" ", sessionId.c_str());
    }
    if (userId) {
        text += Sprintf("s_user_id:\"%s\" ", userId.c_str());
    }
    if (trackQuery.IMEI) {
        text += Sprintf("s_imei:\"%s\" ", trackQuery.IMEI.c_str());
    }
    if (since) {
        text += Sprintf("i_ts_finish:>=%lu ", since.Seconds());
    }
    if (until && until != TInstant::Max()) {
        text += Sprintf("i_ts_start:<=%lu ", until.Seconds());
    }

    NRTLine::TQuery query;
    query.SetText(text);
    query.HowBuilder().SetAttrName("i_ts_start");
    query.HowBuilder().SetSortMethod(NRTLine::TSortPolicy::smAttr);
    query.GroupingBuilder().SortMethodInGroupBuilder().SetAttrName("i_ts_start");
    query.GroupingBuilder().SortMethodInGroupBuilder().SetSortMethod(NRTLine::TSortPolicy::smAttr);
    query.SetNumDoc(numdoc);
    return query;
}

void NDrive::SerializeDebugInfo(NJson::TJsonValue& result, const NGraph::TRouter::TTimedGeoCoordinates& coordinates, const TSpeedLimitRange& range) {
    result["time"] = range.Time;
    result["length"] = range.Length;
    result["SpeedAverage"] = TSpeedWithUnit::MS(range.SpeedAverage).KmH();
    result["SpeedLimit"] = TSpeedWithUnit::MS(range.SpeedLimit).KmH();
    result["Trace"] = TGeoCoord::SerializeVector(range.Trace);
    SerializePoints(result["Points"], coordinates, range.Indices);
}

void NDrive::SerializeMetaInfo(NJson::TJsonValue& result, const TTrack& track) {
    result["car"] = track.DeviceId;
    result["ride"] = track.Name;
    result["user"] = track.UserId;
    result["session"] = track.SessionId;
    result["status"] = ToString(track.Status);
    if (!track.Coordinates.empty()) {
        result["start_timestamp"] = track.Coordinates.front().Timestamp.Seconds();
        result["finish_timestamp"] = track.Coordinates.back().Timestamp.Seconds();
    }
}

void NDrive::SerializePoints(NJson::TJsonValue& result, const NGraph::TRouter::TTimedGeoCoordinates& coordinates, const TVector<size_t>& indices) {
    for (auto&& index : indices) {
        Y_ENSURE(index < coordinates.size());
        const auto& coordinate = coordinates[index];
        auto& value = result.AppendValue(NJson::JSON_ARRAY);
        value.AppendValue(coordinate.Y);
        value.AppendValue(coordinate.X);
        value.AppendValue(coordinate.Timestamp.Seconds());
        value.AppendValue(TSpeedWithUnit::MS(coordinate.Speed).KmH());
    }
}

void NDrive::SerializeViolationIntervals(NJson::TJsonValue& result, const NGraph::TRouter::TTimedGeoCoordinates& coordinates, const TSpeedLimitRange& violation) {
    auto& value = result.AppendValue(NJson::JSON_ARRAY);
    auto polyline = TGeoPolyLine(violation.Projection);
    Y_ENSURE(!violation.Trace.empty());
    value["track"] = SerializeTrace(violation.Projection);
    value["center"] = SerializeCoordinate(polyline.GetCoordByLength(polyline.GetLength() / 2));
    SerializeViolationMeta(value, violation, coordinates);
}
