#pragma once

#include "track.h"
#include "query.h"

#include <rtline/protos/report_accessor.h>

class TRTLineAPIConfig;

namespace NDrive {
    class TTrackVisitor: public TReportAccessor::IVisitor {
    public:
        TTrackVisitor(TString rideId = {}, TMaybe<NDrive::ECarStatus> status = {}, TInstant since = TInstant::Zero(), TInstant until = TInstant::Max())
            : RideId(rideId)
            , Status(status)
            , Since(since)
            , Until(until)
        {
        }

        const TTracks& GetTracks() const {
            return Tracks;
        }
        TTracks ExtractTracks() {
            return std::move(Tracks);
        }

        void SetRideId(const TString& value) {
            RideId = value;
        }
        void SetStatus(NDrive::ECarStatus value) {
            Status = value;
        }
        void SetSince(TInstant value) {
            Since = value;
        }
        void SetUntil(TInstant value) {
            Until = value;
        }

        virtual TReportAccessor::EStatus VisitDoc(const TGrouping& /*grouping*/, const TGroup& /*group*/, const TDocument& doc);

    private:
        TString RideId;
        TMaybe<NDrive::ECarStatus> Status;
        TInstant Since;
        TInstant Until;

        TTracks Tracks;
    };

    template <class T>
    void SerializeCoordinate(NJson::TJsonValue& result, const T& coordinate) {
        result.AppendValue(coordinate.X);
        result.AppendValue(coordinate.Y);
    }

    template <class T>
    NJson::TJsonValue SerializeCoordinate(const T& coordinate) {
        NJson::TJsonValue result(NJson::JSON_ARRAY);
        SerializeCoordinate(result, coordinate);
        return result;
    }

    template <class TTrace>
    void SerializeTrace(NJson::TJsonValue& result, const TTrace& trace) {
        for (auto&& coordinate : trace) {
            SerializeCoordinate(result, coordinate);
        }
    }

    template <class TTrace>
    NJson::TJsonValue SerializeTrace(const TTrace& trace) {
        NJson::TJsonValue result(NJson::JSON_ARRAY);
        SerializeTrace(result, trace);
        return result;
    }

    NRTLine::TQuery MakeTrackQuery(const TTrackQuery& trackQuery);
    NRTLine::TQuery MakeTrackQuery(
        const TString& deviceId, const TString& rideId, const TString& userId, const TString& sessionId,
        TInstant since = TInstant::Zero(), TInstant until = TInstant::Max(), ui32 numdoc = 1000
    );

    void SerializeDebugInfo(NJson::TJsonValue& result, const NGraph::TRouter::TTimedGeoCoordinates& coordinates, const TSpeedLimitRange& range);
    void SerializeMetaInfo(NJson::TJsonValue& result, const TTrack& track);
    void SerializePoints(NJson::TJsonValue& result, const NGraph::TRouter::TTimedGeoCoordinates& coordinates, const TVector<size_t>& indices);
    void SerializeViolationIntervals(NJson::TJsonValue& result, const NGraph::TRouter::TTimedGeoCoordinates& coordinates, const TSpeedLimitRange& violation);
}
