#include "track.h"

#include <rtline/library/json/cast.h>

#include <util/generic/adaptor.h>

TDuration NDrive::CalcDuration(const TTrack& track) {
    if (track.Coordinates.empty()) {
        return TDuration::Zero();
    }
    return track.Coordinates.back().Timestamp - track.Coordinates.front().Timestamp;
}

double NDrive::CalcLength(const TTrack& track) {
    double result = 0;
    for (size_t i = 0; i + 1 < track.Coordinates.size(); ++i) {
        result += track.Coordinates[i].GetLengthTo(track.Coordinates[i + 1]);
    }
    return result;
}

NDrive::TTrack NDrive::CropTrack(TTrack&& track, TInstant since, TInstant until) {
    NGraph::TRouter::TTimedGeoCoordinates coordinates;
    for (auto&& c : track.Coordinates) {
        if (c.Timestamp <= until && c.Timestamp >= since) {
            coordinates.push_back(c);
        }
    }
    track.Coordinates = std::move(coordinates);
    return track;
}

NDrive::TTrack NDrive::CropTrack(const TTrack& track, TInstant since, TInstant until) {
    TTrack resultTrack = track;
    return CropTrack(std::move(resultTrack), since, until);
}

NDrive::TOptionalTrack NDrive::GetTail(const TTracks& tracks, double length) {
    if (tracks.empty()) {
        return {};
    }

    NDrive::TTrack result;
    double preliminaryLength = 0;
    for (auto&& track : tracks) {
        result.Name = track.Name;
        result.DeviceId = track.DeviceId;
        result.SessionId = track.SessionId;
        result.UserId = track.UserId;
        for (auto&& point : Reversed(track.Coordinates)) {
            if (!result.Coordinates.empty()) {
                preliminaryLength += result.Coordinates.back().GetLengthTo(point);
            }
            result.Coordinates.push_back(point);
            if (preliminaryLength >= length) {
                break;
            }
        }
        if (preliminaryLength >= length) {
            break;
        }
    }
    std::reverse(result.Coordinates.begin(), result.Coordinates.end());
    return result;
}

NDrive::TTrackPartition NDrive::BuildSimplePartition(const TInterval<TInstant>& ridingInterval, const NDrive::TTrack& track, const TTrackPartitionOptions& options) {
    NDrive::TTrackPartition result;
    TMaybe<TInstant> start;
    TMaybe<TInstant> finish;
    for (auto&& point : track.Coordinates) {
        if (point.Speed >= options.SpeedThreshold) {
            if (!start) {
                start = point.Timestamp;
            }
        } else {
            if (start) {
                finish = point.Timestamp;
            }
        }
        if (start && finish) {
            result.Intervals.push_back(MakeIntervalRobust<TInstant>(*start, *finish));
            start.Clear();
            finish.Clear();
        }
    }
    if (start) {
        result.Intervals.push_back(MakeIntervalRobust<TInstant>(*start, ridingInterval.GetMax()));
    }
    return result;
}

void NDrive::BuildTrackPartition(const TInterval<TInstant>& ridingInterval, const TTrack& track, const TTrackPartitionOptions& options, TVector<TInterval<TInstant>>& result) {
    result = BuildTrackPartition(ridingInterval, track, options).Intervals;
}

NDrive::TTrackPartition NDrive::BuildTrackPartition(const TInterval<TInstant>& ridingInterval, const TTrack& track, const TTrackPartitionOptions& options) {
    auto fault = options.Fault;
    auto threshold = options.Threshold;

    auto partition = NDrive::TTrackPartition();
    auto& result = partition.Intervals;
    if (track.Coordinates.empty()) {
        result.push_back(ridingInterval);
        return partition;
    }

    auto head = MakeIntervalRobust<TInstant>(track.Coordinates.front().Timestamp, ridingInterval.GetMin());
    if (head.GetMax() - head.GetMin() > fault) {
        partition.Head = head;
        result.push_back(head);
    }

    auto tail = MakeIntervalRobust<TInstant>(track.Coordinates.back().Timestamp, ridingInterval.GetMax());
    if (tail.GetMax() - tail.GetMin() > fault) {
        partition.Tail = tail;
        result.push_back(tail);
    }

    TInstant ridingStart = head.GetMax();
    if (options.AlwaysCutEdges) {
        for (auto&& point : track.Coordinates) {
            if (point.Speed >= options.SpeedThreshold) {
                ridingStart = std::max(ridingStart, point.Timestamp);
                break;
            }
        }
    }

    TMaybe<TInstant> parkingStart;
    for (auto&& point : track.Coordinates) {
        if (point.Timestamp < ridingStart) {
            continue;
        }
        if (point.Speed < options.SpeedThreshold) {
            if (!parkingStart) {
                parkingStart = point.Timestamp;
            }
            continue;
        }

        if (!parkingStart) {
            continue;
        }

        if (point.Timestamp - parkingStart.GetRef() >= threshold) {
            auto interval = MakeIntervalRobust<TInstant>(ridingStart, parkingStart.GetRef());
            if (interval.GetMax() - interval.GetMin() > fault) {
                result.push_back(interval);
                ridingStart = point.Timestamp;
            }
        }
        parkingStart = Nothing();
    }

    auto finish = track.Coordinates.back().Timestamp;
    if (options.AlwaysCutEdges) {
        finish = std::min(finish, ridingInterval.GetMax());
    }
    if (!parkingStart) {
        result.push_back(MakeIntervalRobust<TInstant>(ridingStart, finish));
    } else {
        if (finish - parkingStart.GetRef() >= threshold || options.AlwaysCutEdges) {
            if (ridingStart != parkingStart.GetRef() || !options.SkipEmpty) {
                result.push_back(MakeIntervalRobust<TInstant>(ridingStart, parkingStart.GetRef()));
            }
        } else {
            result.push_back(MakeIntervalRobust<TInstant>(ridingStart, finish));
        }
    }
    return partition;
}

TDuration NDrive::GetDuration(const NDrive::TTrackPartition& partition) {
    TDuration result;
    for (auto&& interval : partition.Intervals) {
        result += (interval.GetMax() - interval.GetMin());
    }
    return result;
}

NDrive::TOptionalTrack NDrive::GetTail(const NDrive::TTracks& tracks, TDuration duration) {
    for (auto&& track : tracks) {
        if (!track.Coordinates.empty()) {
            TInstant until = track.Coordinates.back().Timestamp;
            TInstant since = until - duration;
            return GetTail(tracks, since);
        }
    }
    return {};
}

NDrive::TOptionalTrack NDrive::GetTail(const NDrive::TTracks& tracks, TInstant since) {
    if (tracks.empty()) {
        return {};
    }

    NDrive::TTrack result;
    for (auto&& track : tracks) {
        result.Name = track.Name;
        result.DeviceId = track.DeviceId;
        result.SessionId = track.SessionId;
        result.UserId = track.UserId;
        for (auto&& point : Reversed(track.Coordinates)) {
            if (point.Timestamp < since) {
                break;
            }
            result.Coordinates.push_back(point);
        }
    }
    std::reverse(result.Coordinates.begin(), result.Coordinates.end());
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const TInterval<TInstant>& object) {
    return NJson::ToJson(object.GetPair());
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TTrack& object) {
    NJson::TJsonValue result;
    result["name"] = object.Name;
    result["status"] = ToString(object.Status);
    result["coordinates"] = NJson::ToJson(object.Coordinates);
    result["linked"] = NJson::ToJson(object.Linked);
    if (object.DeviceId) {
        result["device_id"] = object.DeviceId;
    }
    if (object.UserId) {
        result["user_id"] = object.UserId;
    }
    if (object.SessionId) {
        result["session_id"] = object.SessionId;
    }
    if (object.Source) {
        result["source"] = object.Source;
    }
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TTrackPartition& object) {
    NJson::TJsonValue result;
    result["intervals"] = NJson::ToJson(object.Intervals);
    result["head"] = NJson::ToJson(object.Head);
    result["tail"] = NJson::ToJson(object.Tail);
    return result;
}
