#include "projector.h"

#include <library/cpp/testing/unittest/registar.h>

namespace {
    using TTrack = NDrive::TTrack;
    using TTracks = NDrive::TTracks;

    using TLinkedTrack = NDrive::TTracksLinker::TResult;
    using TLinkedTracks = NDrive::TTracksLinker::TResults;

    TInstant Ticks(int m) {
        return TInstant::Zero() + TDuration::MicroSeconds(m);
    }

    NGraph::NRouter::TTimedGeoCoordinates BuildCoords(const TRange<TInstant>& range, const TDuration step = TDuration::MicroSeconds(1)) {
        int coord = 0;
        TInstant time = *range.From;
        NGraph::NRouter::TTimedGeoCoordinates coords;
        while (time <= *range.To) {
            coords.push_back({{coord, coord}, time});
            ++coord;
            time += step;
        }
        return coords;
    }

    TTrack BuildTrack(const TRange<TInstant>& range, const TDuration& step = TDuration::MicroSeconds(1)) {
        TTrack track;
        track.Coordinates = BuildCoords(range, step);
        return track;
    }

    TLinkedTrack BuildLinkedTrack(const TRange<TInstant>& range, const TDuration& step = TDuration::MicroSeconds(1)) {
        TLinkedTrack linkedTrack;
        linkedTrack.FilteredCoordinates = BuildCoords(range, step);
        return linkedTrack;
    }
}

Y_UNIT_TEST_SUITE(TracksOps) {
    Y_UNIT_TEST(SubtractLinked) {
        {
            TTracks tracks;
            tracks.push_back(BuildTrack({Ticks(0), Ticks(2)}));
            tracks.push_back(BuildTrack({Ticks(6), Ticks(8)}));

            TLinkedTracks linkedTracks;
            linkedTracks.push_back(BuildLinkedTrack({Ticks(3), Ticks(5)}));

            const auto subtractedTracks = NDrive::TTrackOps::SubtractLinked(tracks, linkedTracks);
            UNIT_ASSERT_EQUAL(subtractedTracks.size(), 2);
        }

        {
            TTracks tracks;
            tracks.push_back(BuildTrack({Ticks(0), Ticks(10)}));

            TLinkedTracks linkedTracks;
            linkedTracks.push_back(BuildLinkedTrack({Ticks(4), Ticks(6)}));

            const auto subtractedTracks = NDrive::TTrackOps::SubtractLinked(tracks, linkedTracks);
            UNIT_ASSERT_EQUAL(subtractedTracks.size(), 2);
            UNIT_ASSERT_EQUAL(subtractedTracks[0].Coordinates.front().Timestamp, Ticks(0));
            UNIT_ASSERT_EQUAL(subtractedTracks[1].Coordinates.front().Timestamp, Ticks(6));
        }

        {
            TTracks tracks;
            tracks.push_back(BuildTrack({Ticks(0), Ticks(10)}));

            TLinkedTracks linkedTracks;
            linkedTracks.push_back(BuildLinkedTrack({Ticks(4), Ticks(6)}));
            linkedTracks.push_back(BuildLinkedTrack({Ticks(5), Ticks(7)}));

            const auto subtractedTracks = NDrive::TTrackOps::SubtractLinked(tracks, linkedTracks);
            UNIT_ASSERT_EQUAL(subtractedTracks.size(), 2);
            UNIT_ASSERT_EQUAL(subtractedTracks[0].Coordinates.front().Timestamp, Ticks(0));
            UNIT_ASSERT_EQUAL(subtractedTracks[1].Coordinates.front().Timestamp, Ticks(7));
        }

        {
            TTracks tracks;
            tracks.push_back(BuildTrack({Ticks(0), Ticks(15)}));

            TLinkedTracks linkedTracks;
            linkedTracks.push_back(BuildLinkedTrack({Ticks(4), Ticks(6)}));
            linkedTracks.push_back(BuildLinkedTrack({Ticks(9), Ticks(12)}));

            const auto subtractedTracks = NDrive::TTrackOps::SubtractLinked(tracks, linkedTracks);
            UNIT_ASSERT_EQUAL(subtractedTracks.size(), 3);
            UNIT_ASSERT_EQUAL(subtractedTracks[0].Coordinates.front().Timestamp, Ticks(0));
            UNIT_ASSERT_EQUAL(subtractedTracks[1].Coordinates.front().Timestamp, Ticks(6));
            UNIT_ASSERT_EQUAL(subtractedTracks[2].Coordinates.front().Timestamp, Ticks(12));
        }

        {
            TTracks tracks;
            tracks.push_back(BuildTrack({Ticks(0), Ticks(4)}));
            tracks.push_back(BuildTrack({Ticks(10), Ticks(15)}));

            TLinkedTracks linkedTracks;
            linkedTracks.push_back(BuildLinkedTrack({Ticks(2), Ticks(12)}));

            const auto subtractedTracks = NDrive::TTrackOps::SubtractLinked(tracks, linkedTracks);
            UNIT_ASSERT_EQUAL(subtractedTracks.size(), 2);
            UNIT_ASSERT_EQUAL(subtractedTracks[0].Coordinates.front().Timestamp, Ticks(0));
            UNIT_ASSERT_EQUAL(subtractedTracks[0].Coordinates.back().Timestamp, Ticks(2));
            UNIT_ASSERT_EQUAL(subtractedTracks[1].Coordinates.front().Timestamp, Ticks(12));
            UNIT_ASSERT_EQUAL(subtractedTracks[1].Coordinates.back().Timestamp, Ticks(15));
        }

        {
            TTracks tracks;
            tracks.push_back(BuildTrack({Ticks(0), Ticks(10)}));

            TLinkedTracks linkedTracks;
            linkedTracks.push_back(BuildLinkedTrack({Ticks(1), Ticks(2)}));
            linkedTracks.push_back(BuildLinkedTrack({Ticks(3), Ticks(4)}));
            linkedTracks.push_back(BuildLinkedTrack({Ticks(5), Ticks(6)}));
            linkedTracks.push_back(BuildLinkedTrack({Ticks(7), Ticks(8)}));
            linkedTracks.push_back(BuildLinkedTrack({Ticks(9), Ticks(10)}));

            const auto subtractedTracks = NDrive::TTrackOps::SubtractLinked(tracks, linkedTracks);
            UNIT_ASSERT_EQUAL(subtractedTracks.size(), 6);

            UNIT_ASSERT_EQUAL(subtractedTracks[0].Coordinates.front().Timestamp, Ticks(0));
            UNIT_ASSERT_EQUAL(subtractedTracks[0].Coordinates.back().Timestamp, Ticks(1));

            UNIT_ASSERT_EQUAL(subtractedTracks[5].Coordinates.front().Timestamp, Ticks(10));
            UNIT_ASSERT_EQUAL(subtractedTracks[5].Coordinates.back().Timestamp, Ticks(10));
        }

        {
            TTracks tracks;
            tracks.push_back(BuildTrack({Ticks(0), Ticks(4)}));

            TLinkedTracks linkedTracks;
            linkedTracks.push_back(BuildLinkedTrack({Ticks(2), Ticks(2)}));

            const auto subtractedTracks = NDrive::TTrackOps::SubtractLinked(tracks, linkedTracks);
            UNIT_ASSERT_EQUAL(subtractedTracks.size(), 2);
            UNIT_ASSERT_EQUAL(subtractedTracks[0].Coordinates.front().Timestamp, Ticks(0));
            UNIT_ASSERT_EQUAL(subtractedTracks[0].Coordinates.back().Timestamp, Ticks(2));
            UNIT_ASSERT_EQUAL(subtractedTracks[1].Coordinates.front().Timestamp, Ticks(2));
            UNIT_ASSERT_EQUAL(subtractedTracks[1].Coordinates.back().Timestamp, Ticks(4));
        }

    }
}
