#include "client.h"

#include <library/cpp/mediator/global_notifications/system_status.h>
#include <library/cpp/json/json_reader.h>
#include <rtline/library/unistat/cache.h>
#include <rtline/library/unistat/signals.h>
#include <rtline/library/json/parse.h>
#include <rtline/library/json/builder.h>

void TTrustClientConfig::Init(const TYandexConfig::Section* section) {
    Name = section->GetDirectives().Value("Name", Name);
    AssertCorrectConfig(!!Name, "incorrect service name");
    Host = section->GetDirectives().Value("Host", Host);
    AssertCorrectConfig(!!Host, "incorrect trust host");
    RequestTimeout = section->GetDirectives().Value("RequestTimeout", RequestTimeout);

    SelfTvmId = section->GetDirectives().Value("SelfTvmId", SelfTvmId);
    DestinationTvmId = section->GetDirectives().Value("DestinationTvmId", DestinationTvmId);

    const TYandexConfig::TSectionsMap children = section->GetAllChildren();
    auto it = children.find("RequestConfig");
    if (it != children.end()) {
        RequestConfig.InitFromSection(it->second);
    }

    PaymentPath = section->GetDirectives().Value("PaymentPath", PaymentPath);
    PaymentInfoPath = section->GetDirectives().Value("PaymentInfoPath", PaymentInfoPath);
    PaymentMethodsPath = section->GetDirectives().Value("PaymentMethodsPath", PaymentMethodsPath);
    RefundsPath = section->GetDirectives().Value("RefundsPath", RefundsPath);
    YAccountPath = section->GetDirectives().Value("YAccountPath", YAccountPath);
    TopupPath = section->GetDirectives().Value("TopupPath", TopupPath);
}

void TTrustClientConfig::ToString(IOutputStream& os) const {
    os << "Name: " << Name << Endl;
    os << "Host: " << Host << Endl;
    if (SelfTvmId) {
        os << "SelfTvmId: " << SelfTvmId << Endl;
        os << "DestinationTvmId: " << DestinationTvmId << Endl;
    }

    os << "RequestTimeout: " << RequestTimeout << Endl;
    os << "<RequestConfig>" << Endl;
    RequestConfig.ToString(os);
    os << "</RequestConfig>" << Endl;
    os << "PaymentPath: " << PaymentPath << Endl;
    os << "PaymentInfoPath: " << PaymentInfoPath << Endl;
    os << "PaymentMethodsPath: " << PaymentMethodsPath << Endl;
    os << "RefundsPath: " << RefundsPath << Endl;
    os << "YAccountPath: " << YAccountPath << Endl;
    os << "TopupPath: " << TopupPath << Endl;
}

TTrustClient::TTrustClient(const TTrustClientConfig& config, TAtomicSharedPtr<NTvmAuth::TTvmClient> tvmClient)
    : Config(config)
    , TvmClient(tvmClient)
{
    Agent = MakeHolder<NNeh::THttpClient>(Config.GetHost(), Config.GetRequestConfig());
}

NThreading::TFuture<TVector<NDrive::NTrustClient::TPaymentMethod>> TTrustClient::GetPaymentMethods(const TString& serviceToken, const TString& passportId) const {
    NNeh::THttpRequest request;
    request.AddHeader("X-Uid", passportId);
    request.SetUri(Config.GetPaymentMethodsPath());
    return SendRequest(std::move(request), serviceToken).Apply([](const NThreading::TFuture<NJson::TJsonValue>& r) {
        auto json = r.GetValue();
        TVector<NDrive::NTrustClient::TPaymentMethod> results;
        for (auto&& payment : json["bound_payment_methods"].GetArray()) {
            NDrive::NTrustClient::TPaymentMethod method;
            TMessagesCollector errors;
            if (!method.FromJson(payment, &errors)) {
                throw yexception() << "cannot parse: " << payment.GetStringRobust() << " " << errors.GetStringReport();
            }
            results.push_back(method);
        }
        return results;
    });
}

NThreading::TFuture<TString> TTrustClient::CreatePayment(const TString& serviceToken, const TString& passportId, const NDrive::NTrustClient::TPayment& payment) const {
    NJson::TJsonValue postData;
    payment.ToJson(postData);

    NNeh::THttpRequest request;
    request.SetPostData(postData);
    request.AddHeader("X-Uid", passportId);
    request.SetUri(Config.GetPaymentPath());
    return SendRequest(std::move(request), serviceToken).Apply([](const NThreading::TFuture<NJson::TJsonValue>& r) {
        auto json = r.GetValue();
        TString paymentId = json["purchase_token"].GetString();
        if (paymentId.empty()) {
            throw yexception() << "empty purchase_token, report: " << json.GetStringRobust();
        }
        return paymentId;
    });
}
NThreading::TFuture<NDrive::NTrustClient::TPayment> TTrustClient::StartPayment(const TString& serviceToken, const TString& purchaseToken) const {
    NNeh::THttpRequest request;
    request.SetRequestType("POST");
    request.SetUri(Config.GetPaymentPath()+ "/" + purchaseToken + "/start");
    return SendRequest(std::move(request), serviceToken).Apply([](const NThreading::TFuture<NJson::TJsonValue>& r) {
        auto json = r.GetValue();
        NDrive::NTrustClient::TPayment payment;
        payment.FromJson(json);
        return payment;
    });
}
NThreading::TFuture<NDrive::NTrustClient::TPayment> TTrustClient::GetPayment(const TString& serviceToken, const TString& purchaseToken, bool getProcessingInfo) const {
    NNeh::THttpRequest request;
    request.SetUri(Config.GetPaymentInfoPath() + "/" + purchaseToken);
    if (getProcessingInfo) {
        request.AddCgiData("&show_processing_info=true");
    }
    return SendRequest(std::move(request), serviceToken).Apply([](const NThreading::TFuture<NJson::TJsonValue>& r) {
        auto json = r.GetValue();
        NDrive::NTrustClient::TPayment payment;
        payment.FromJson(json);
        return payment;
    });
}
NThreading::TFuture<TString> TTrustClient::CancelPayment(const TString& serviceToken, const TString& purchaseToken) const {
    NNeh::THttpRequest request;
    request.SetRequestType("POST");
    request.SetUri(Config.GetPaymentPath()+ "/" + purchaseToken + "/unhold");
    return SendRequest(std::move(request), serviceToken).Apply([](const NThreading::TFuture<NJson::TJsonValue>& r) {
        auto json = r.GetValue();
        return json["status_code"].GetString();
    });
}
NThreading::TFuture<TString> TTrustClient::ClearPayment(const TString& serviceToken, const TString& purchaseToken) const {
    NNeh::THttpRequest request;
    request.SetRequestType("POST");
    request.SetUri(Config.GetPaymentPath()+ "/" + purchaseToken + "/clear");
    return SendRequest(std::move(request), serviceToken).Apply([](const NThreading::TFuture<NJson::TJsonValue>& r) {
        auto json = r.GetValue();
        return json["status_code"].GetString();
    });
}
NThreading::TFuture<void> TTrustClient::ResizePayment(const TString& serviceToken, const TString& purchaseToken, const TString& orderId, NJson::NPrivate::TMoney<const i64> leftSum) const {
    NJson::TJsonValue postData;
    NJson::InsertField(postData, "amount", leftSum);
    NJson::InsertField(postData, "qty", 1);

    NNeh::THttpRequest request;
    request.SetPostData(postData);
    request.SetUri(Config.GetPaymentPath() + "/" + purchaseToken + "/orders/" + orderId + "/resize");
    return SendRequest(std::move(request), serviceToken).IgnoreResult();
}
NThreading::TFuture<TString> TTrustClient::CreateRefund(const TString& serviceToken, const NDrive::NTrustClient::TRefund& refund) const {
    NJson::TJsonValue postData;
    refund.ToJson(postData);

    NNeh::THttpRequest request;
    request.SetPostData(postData);
    request.SetUri(Config.GetRefundsPath());
    return SendRequest(std::move(request), serviceToken).Apply([](const NThreading::TFuture<NJson::TJsonValue>& r) {
        auto json = r.GetValue();
        TString trustRefundId = json["trust_refund_id"].GetString();
        if (!trustRefundId) {
            throw yexception() << "empty trust_refund_id, report: " << json.GetStringRobust();
        }
        return trustRefundId;
    });
}
NThreading::TFuture<NDrive::NTrustClient::TRefundStatus> TTrustClient::StartRefund(const TString& serviceToken, const TString& refundId) const {
    NNeh::THttpRequest request;
    request.SetRequestType("POST");
    request.SetUri(Config.GetRefundsPath() + "/" + refundId + "/start");
    return SendRequest(std::move(request), serviceToken, false).Apply([refundId](const NThreading::TFuture<NJson::TJsonValue>& r) {
        auto json = r.GetValue();
        NDrive::NTrustClient::TRefundStatus status;
        if (!NJson::ParseField(json, status)) {
            throw yexception() << "cannot parse refund " + refundId + " status: " << json.GetStringRobust();
        }
        return status;
    });
}
NThreading::TFuture<NDrive::NTrustClient::TRefundStatus> TTrustClient::GetRefund(const TString& serviceToken, const TString& refundId) const {
    NNeh::THttpRequest request;
    request.SetUri(Config.GetRefundsPath() + "/" + refundId);
    return SendRequest(std::move(request), serviceToken, false).Apply([refundId](const NThreading::TFuture<NJson::TJsonValue>& r) {
        auto json = r.GetValue();
        NDrive::NTrustClient::TRefundStatus status;
        if (!NJson::ParseField(json, status)) {
            throw yexception() << "cannot parse refund " + refundId + " status: " << json.GetStringRobust();
        }
        return status;
    });
}
NThreading::TFuture<TString> TTrustClient::CreateTopup(const TString& serviceToken, const TString& passportId, const NDrive::NTrustClient::TPayment& payment) const {
    NJson::TJsonValue postData;
    payment.ToJson(postData);

    NNeh::THttpRequest request;
    request.SetPostData(postData);
    request.SetUri(Config.GetTopupPath());
    request.AddHeader("X-Uid", passportId);
    return SendRequest(std::move(request), serviceToken).Apply([](const NThreading::TFuture<NJson::TJsonValue>& r) {
        auto json = r.GetValue();
        TString paymentId = json["purchase_token"].GetString();
        if (paymentId.empty()) {
            throw yexception() << "empty purchase_token, report: " << json.GetStringRobust();
        }
        return paymentId;
    });
}
NThreading::TFuture<NDrive::NTrustClient::TPayment> TTrustClient::StartTopup(const TString& serviceToken, const TString& purchaseToken) const {
    NNeh::THttpRequest request;
    request.SetRequestType("POST");
    request.SetUri(Config.GetTopupPath()+ "/" + purchaseToken + "/start");
    return SendRequest(std::move(request), serviceToken).Apply([](const NThreading::TFuture<NJson::TJsonValue>& r) {
        auto json = r.GetValue();
        NDrive::NTrustClient::TPayment payment;
        payment.FromJson(json);
        return payment;
    });
}
NThreading::TFuture<NDrive::NTrustClient::TPayment> TTrustClient::TopupInfo(const TString& serviceToken, const TString& purchaseToken) const {
    NNeh::THttpRequest request;
    request.SetUri(Config.GetTopupPath() + "/" + purchaseToken);
    return SendRequest(std::move(request), serviceToken).Apply([](const NThreading::TFuture<NJson::TJsonValue>& r) {
        auto json = r.GetValue();
        NDrive::NTrustClient::TPayment payment;
        payment.FromJson(json);
        return payment;
    });
}
NThreading::TFuture<void> TTrustClient::CreateYAccount(const TString& serviceToken, const TString& passportId, const TString& currency) const {
    NNeh::THttpRequest request;
    request.SetPostData(NJson::TMapBuilder("currency", currency));
    request.SetUri(Config.GetYAccountPath());
    request.AddHeader("X-Uid", passportId);
    return SendRequest(std::move(request), serviceToken).IgnoreResult();
}

NThreading::TFuture<NJson::TJsonValue> TTrustClient::SendRequest(NNeh::THttpRequest&& request, const TString& serviceToken, bool checkStatus) const {
    if (TvmClient) {
        request.AddHeader("X-Ya-Service-Ticket", TvmClient->GetServiceTicketFor(Config.GetDestinationTvmId()));
    }
    request.AddHeader("X-Service-Token", serviceToken);

    auto startTs = Now();
    return Agent->SendAsync(request, Now() + Config.GetRequestTimeout()).Apply([startTs, checkStatus, request = std::move(request), service = Config.GetName()](const NThreading::TFuture<NNeh::THttpReply>& r) -> NJson::TJsonValue {
        auto& uri = request.GetUri();
        auto& method = request.GetRequestType();
        TUnistatSignalsCache::SignalAdd(service, "trust-requests-" + uri + "-" + method, 1);
        auto duration = Now() - startTs;
        if (!r.HasValue()) {
            TUnistatSignalsCache::SignalAdd(service, "trust-reply-codes-" + uri + "-" + method + "-timeout", 1);
            TUnistatSignalsCache::SignalHistogram(service, "trust-times-" + uri + "-" + method + "-timeout", duration.MilliSeconds(), NRTLineHistogramSignals::IntervalsRTLineReply);
            throw yexception() << "timeout: " << duration;
        }
        auto result = r.GetValue();
        TUnistatSignalsCache::SignalAdd(service, "trust-reply-codes-" + uri + "-" + method + "-" + ToString((ui32)result.Code()), 1);
        TUnistatSignalsCache::SignalHistogram(service, "trust-times-" + uri + "-" + method + "-" + ToString((ui32)result.Code()), duration.MilliSeconds(), NRTLineHistogramSignals::IntervalsRTLineReply);

        NJson::TJsonValue json;
        if (result.Code() != HTTP_OK || !NJson::ReadJsonFastTree(result.Content(), &json)) {
            throw yexception() << "incorrect response: " << result.Code() << " " << result.Content();
        };
        if (checkStatus && json["status"].GetString() != "success") {
            throw yexception() << "error: " << json.GetStringRobust();
        }
        return json;
    });
}
