#pragma once
#include "entity.h"
#include <drive/library/cpp/threading/future.h>

#include <library/cpp/tvmauth/client/facade.h>

#include <rtline/util/network/neh.h>
#include <rtline/util/types/accessor.h>

class TTrustClientConfig {
public:
    R_READONLY(TString, Host);
    R_READONLY(TDuration, RequestTimeout, TDuration::Seconds(60));
    R_READONLY(TString, Name);

    R_READONLY(ui64, SelfTvmId, 0);
    R_READONLY(ui64, DestinationTvmId, 0);

    R_FIELD(NSimpleMeta::TConfig, RequestConfig);

    R_READONLY(TString, PaymentPath, "trust-payments/v2/payments");
    R_READONLY(TString, PaymentInfoPath, "trust-payments/v2/payments");
    R_READONLY(TString, PaymentMethodsPath, "trust-payments/v2/payment-methods");
    R_READONLY(TString, RefundsPath, "trust-payments/v2/refunds");
    R_READONLY(TString, YAccountPath, "trust-payments/v2/account");
    R_READONLY(TString, TopupPath, "trust-payments/v2/topup");

public:
    void Init(const TYandexConfig::Section* section);
    void ToString(IOutputStream& os) const;

    static TTrustClientConfig ParseFromString(const TString& configStr) {
        TTrustClientConfig result;
        TAnyYandexConfig config;
        CHECK_WITH_LOG(config.ParseMemory(configStr.data()));
        result.Init(config.GetRootSection());
        return result;
    }
};

class TTrustClient {

public:
    TTrustClient(const TTrustClientConfig& config, TAtomicSharedPtr<NTvmAuth::TTvmClient> tvmClient = nullptr);

    NThreading::TFuture<TVector<NDrive::NTrustClient::TPaymentMethod>> GetPaymentMethods(const TString& serviceToken, const TString& passportId) const;

    NThreading::TFuture<TString> CreatePayment(const TString& serviceToken, const TString& passportId, const NDrive::NTrustClient::TPayment& payment) const;
    NThreading::TFuture<NDrive::NTrustClient::TPayment> StartPayment(const TString& serviceToken, const TString& purchaseToken) const;
    NThreading::TFuture<NDrive::NTrustClient::TPayment> GetPayment(const TString& serviceToken, const TString& purchaseToken, bool getProcessingInfo = false) const;
    NThreading::TFuture<TString> CancelPayment(const TString& serviceToken, const TString& purchaseToken) const;
    NThreading::TFuture<TString> ClearPayment(const TString& serviceToken, const TString& purchaseToken) const;
    NThreading::TFuture<void> ResizePayment(const TString& serviceToken, const TString& purchaseToken, const TString& orderId, NJson::NPrivate::TMoney<const i64> leftSum) const;

    NThreading::TFuture<TString> CreateRefund(const TString& serviceToken, const NDrive::NTrustClient::TRefund& refund) const;
    NThreading::TFuture<NDrive::NTrustClient::TRefundStatus> StartRefund(const TString& serviceToken, const TString& refundId) const;
    NThreading::TFuture<NDrive::NTrustClient::TRefundStatus> GetRefund(const TString& serviceToken, const TString& refundId) const;

    NThreading::TFuture<TString> CreateTopup(const TString& serviceToken, const TString& passportId, const NDrive::NTrustClient::TPayment& payment) const;
    NThreading::TFuture<NDrive::NTrustClient::TPayment> StartTopup(const TString& serviceToken, const TString& purchaseToken) const;
    NThreading::TFuture<NDrive::NTrustClient::TPayment> TopupInfo(const TString& serviceToken, const TString& purchaseToken) const;

    NThreading::TFuture<void> CreateYAccount(const TString& serviceToken, const TString& passportId, const TString& currency) const;

private:
    NThreading::TFuture<NJson::TJsonValue> SendRequest(NNeh::THttpRequest&& request, const TString& serviceToken, bool checkStatus = true) const;

private:
    THolder<NNeh::THttpClient> Agent;
    const TTrustClientConfig Config;
    TAtomicSharedPtr<NTvmAuth::TTvmClient> TvmClient;
};
