#include "entity.h"

#include <rtline/library/json/parse.h>
#include <rtline/library/json/adapters.h>
#include <rtline/library/json/merge.h>
#include <util/string/vector.h>

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NJson::NPrivate::TMoney<i64>& result) {
    double balance = 0;
    if (!TryFromJson(value, balance)) {
        return false;
    }
    result.Value = std::round(balance * 100);
    return true;
}

template <>
NJson::TJsonValue NJson::ToJson(const NJson::NPrivate::TMoney<const i64>& result) {
    return ::ToString(result.Value / 100.);
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NTrustClient::TPaymentMethod::TPayerInfo::TFamilyInfo& result) {
    return
        NJson::ParseField(value, "currency", result.MutableCurrency()) &&
        NJson::ParseField(value, "frame", result.MutableFrame()) &&
        NJson::ParseField(value, "limit", result.MutableLimit(), false) &&
        NJson::ParseField(value, "expenses", result.MutableExpenses(), false) &&
        NJson::ParseField(value, "family_id", result.MutableFamilyId(), true);
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NTrustClient::TPaymentMethod::TPayerInfo::TFamilyInfo& object) {
    NJson::TJsonValue result;
    NJson::InsertField(result, "currency", object.GetCurrency());
    NJson::InsertField(result, "frame", object.GetFrame());
    NJson::InsertField(result, "limit", object.GetLimit());
    NJson::InsertField(result, "expenses", object.GetExpenses());
    NJson::InsertField(result, "family_id", object.GetFamilyId());
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NTrustClient::TPaymentMethod::TPayerInfo& result) {
    return
        NJson::ParseField(value, "uid", result.MutableUid()) &&
        NJson::ParseField(value, "family_info", result.OptionalFamilyInfo());
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NTrustClient::TPaymentMethod::TPayerInfo& object) {
    NJson::TJsonValue result;
    NJson::InsertNonNull(result, "uid", object.GetUid());
    NJson::InsertNonNull(result, "family_info", object.OptionalFamilyInfo());
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NTrustClient::TPaymentMethod::TPartnerInfo& result) {
    return
        NJson::ParseField(value, "is_yabank_card", result.OptionalYaBankCard()) &&
        NJson::ParseField(value, "is_yabank_card_owner", result.OptionalYaBankCardOwner());
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NTrustClient::TPaymentMethod::TPartnerInfo& object) {
    NJson::TJsonValue result;
    NJson::InsertNonNull(result, "is_yabank_card", object.OptionalYaBankCard());
    NJson::InsertNonNull(result, "is_yabank_card_owner", object.OptionalYaBankCardOwner());
    return result;
}

bool NDrive::NTrustClient::TPaymentMethod::FromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    if (!NJson::ParseField(json, "account", Account, errors)) {
        return false;
    }
    TVector<TString> parts = SplitString(Account, "*");
    if (parts.size() == 2) {
        Suffix = parts[1];
        if (!TryFromString<ui32>(parts[0], BIN)) {
            return false;
        }
    }
    double bts = 0;
    if (!NJson::ParseField(json, "binding_ts", bts, false, errors)) {
        return false;
    }
    if (!NJson::ParseField(json, "request_passport_id", Uid, /* required = */ false, errors)) {
        return false;
    }

    if (!NJson::ParseField(json, "payer_info", PayerInfo, errors)) {
        return false;
    }
    if (PayerInfo) {
        if (PayerInfo->OptionalFamilyInfo() && PayerInfo->OptionalFamilyInfo()->GetCurrency()) {
            Currency = PayerInfo->OptionalFamilyInfo()->GetCurrency();
        }
    }
    if (!NJson::ParseField(json, "is_family_card_owner", FamilyCardOwner, false, errors)) {
        return false;
    }
    if (!NJson::ParseField(json, "partner_info", PartnerInfo, errors)) {
        return false;
    }
    Binding = TInstant::Seconds(bts);
    return NJson::ParseField(json, "payment_method", PaymentMethod, errors)
        && NJson::ParseField(json, "id", Id, errors)
        && NJson::ParseField(json, "card_level", CardLevel, errors)
        && NJson::ParseField(json, "card_bank", CardBank, errors)
        && NJson::ParseField(json, "system", PSystem, errors)
        && NJson::ParseField(json, "currency", Currency, errors)
        && NJson::ParseField(json, "expired", IsExpired, false, errors)
        && NJson::ParseField(json, "expiration_year", ExpirationYear, errors)
        && NJson::ParseField(json, "expiration_month", ExpirationMonth, errors)
        && NJson::ParseField(json, "holder", Holder, errors)
        && NJson::ParseField(json, "balance", NJson::Money(Balance), errors)
        && NJson::ParseField(json, "aliases", Aliases, errors);
}

NJson::TJsonValue NDrive::NTrustClient::TPaymentMethod::GetReport() const {
    NJson::TJsonValue report;
    NJson::InsertField(report, "payment_method", PaymentMethod);
    NJson::InsertField(report, "id", Id);
    NJson::InsertField(report, "account", Account);
    NJson::InsertField(report, "system", PSystem);
    NJson::InsertField(report, "card_level", CardLevel);
    NJson::InsertField(report, "card_bank", CardBank);
    NJson::InsertField(report, "binding_ts", NJson::Seconds(Binding));
    NJson::InsertField(report, "balance", Balance);
    NJson::InsertField(report, "currency", Currency);
    NJson::InsertField(report, "expired", IsExpired);
    NJson::InsertNonNull(report, "expiration_year", ExpirationYear);
    NJson::InsertNonNull(report, "expiration_month", ExpirationMonth);
    NJson::InsertField(report, "holder", Holder);
    NJson::InsertField(report, "aliases", Aliases);
    NJson::InsertNonNull(report, "payer_info", PayerInfo);
    NJson::InsertNonNull(report, "is_family_card_owner", FamilyCardOwner);
    NJson::InsertNonNull(report, "partner_info", PartnerInfo);
    return report;
}

NJson::TJsonValue NDrive::NTrustClient::TPaymentMethod::ToJson() const {
    NJson::TJsonValue report;
    NJson::InsertField(report, "payment_method", PaymentMethod);
    NJson::InsertField(report, "id", Id);
    NJson::InsertField(report, "account", Account);
    NJson::InsertField(report, "system", PSystem);
    NJson::InsertField(report, "card_level", CardLevel);
    NJson::InsertField(report, "card_bank", CardBank);
    NJson::InsertField(report, "binding_ts", ToString(Binding.Seconds()));
    NJson::InsertField(report, "balance", NJson::Money(Balance));
    NJson::InsertField(report, "currency", Currency);
    NJson::InsertField(report, "expired", IsExpired);
    NJson::InsertNonNull(report, "expiration_year", ExpirationYear);
    NJson::InsertNonNull(report, "expiration_month", ExpirationMonth);
    NJson::InsertField(report, "holder", Holder);
    NJson::InsertField(report, "aliases", Aliases);
    NJson::InsertNonNull(report, "payer_info", PayerInfo);
    NJson::InsertNonNull(report, "is_family_card_owner", FamilyCardOwner);
    NJson::InsertNonNull(report, "partner_info", PartnerInfo);
    return report;
}

NJson::TJsonValue NDrive::NTrustClient::TPayment::TPayload::ToJson() const {
    NJson::TJsonValue result;
    NJson::InsertNonNull(result, "service_id", ::ToString(ServiceId));
    NJson::InsertNonNull(result, "cashback_type", CashbackType);
    if (BaseAmount) {
        NJson::InsertField(result, "base_amount", ::ToString(BaseAmount * 0.01));
    }
    if (IsPlusUser) {
        NJson::InsertField(result, "has_plus", TString(*IsPlusUser ? "true" : "false"));
    }
    if (PayloadType == "drive") {
        NJson::InsertNonNull(result, "offer_type", OfferType);
        NJson::InsertNonNull(result, "action_id", OfferName);
    } else if (PayloadType == "scooters") {
        NJson::InsertNonNull(result, "order_id", OrderId);
    }
    if (PayloadExtra) {
        return NJson::MergeJson(*PayloadExtra, result);
    }
    return result;
}

NDrive::NTrustClient::TPayment::TTerminalRouteData::TTerminalRouteData(const TString& terminal)
    : Description(terminal)
{
}

NJson::TJsonValue NDrive::NTrustClient::TPayment::TTerminalRouteData::ToJson() const {
    NJson::TJsonValue result;
    NJson::InsertNonNull(result, "description", Description);
    return result;
}

void NDrive::NTrustClient::TPayment::FromJson(const NJson::TJsonValue& json) {
    TryFromString(json["payment_status"].GetString(), PaymentStatus);;
    PaymentError = json["payment_resp_code"].GetString();
    Status = json["status"].GetString();
    PaymentErrorDescription = json["payment_resp_desc"].GetString();
    PurchaseToken = json["purchase_token"].GetString();
    PaymethodId = json["paymethod_id"].GetString();
    Currency = json["currency"].GetString();
    Uid = json["uid"].GetString();
    CardMask = json["user_account"].GetString();
    RRN = json["rrn"].GetString();
    if (json.Has("orders") && json["orders"].IsArray() && json["orders"].GetArray().size()) {
        OrderId = json["orders"][0]["order_id"].GetString();
        ProductId = json["orders"][0]["product_id"].GetString();
    }
    Amount = std::round(100.0 * FromString<double>(json["amount"].GetString()));
    if (json.Has("cleared_amount")) {
        Cleared = std::round(100.0 * FromString<double>(json["cleared_amount"].GetString()));
    }
    auto refunds = json["refunds"].GetArray();
    for (auto refund : refunds) {
        ui32 sum = std::round(100.0 * FromString<double>(refund["amount"].GetString()));
        Refunded += sum;
    }
    if (json.Has("processing_info")) {
        ProcessingInfo = json["processing_info"];
    }
}
void NDrive::NTrustClient::TPayment::ToJson(NJson::TJsonValue& json) const {
    json["amount"] = Amount * 0.01;
    json["currency"] = Currency;
    json["paymethod_id"] = PaymethodId;
    if (!!Email) {
        json["user_email"] = Email;
    }
    if (!!ProductId) {
        json["product_id"] = ProductId;
    }
    if (!!OrderId) {
        json["order_id"] = OrderId;
    }
    if (!!FiscalTitle) {
        json["fiscal_title"] = FiscalTitle;
        json["fiscal_nds"] = FiscalNDS;
    }
    if (Payload) {
        json["pass_params"]["payload"] = Payload->ToJson();
    }
    if (TerminalRouteData) {
        json["pass_params"]["terminal_route_data"] = TerminalRouteData->ToJson();
    }
    if (Uid) {
        json["uid"] = Uid;
    }
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NTrustClient::TRefundStatus& result) {
    return NJson::ParseField(value, "status", result.Code)
        && NJson::ParseField(value, "status_desc", result.Description);
}
