#pragma once

#include <library/cpp/json/json_value.h>
#include <rtline/util/types/accessor.h>
#include <rtline/util/types/messages_collector.h>
#include <util/generic/set.h>

namespace NJson {
    namespace NPrivate {
        template <class T>
        struct TMoney {
        public:
            TMoney(T& value)
                : Value(value)
            {
            }

        public:
            T& Value;
        };
    }

    template <class T>
    NPrivate::TMoney<T> Money(T& object) {
        return { object };
    }

    template <class T>
    bool TryFromJson(const NJson::TJsonValue& value, NPrivate::TMoney<T>&& result) {
        return TryFromJson(value, result);
    }
}

namespace NDrive {
namespace NTrustClient {

class TPaymentMethod {
public:
    class TPayerInfo {
    public:
        class TFamilyInfo {
            R_FIELD(TString, Currency);
            R_FIELD(TString, FamilyId);
            R_FIELD(TString, Frame);
            R_FIELD(i64, Limit, 0);
            R_FIELD(i64, Expenses, 0);
        };
        R_FIELD(TString, Uid);
        R_OPTIONAL(TFamilyInfo, FamilyInfo);
    };

    class TPartnerInfo {
    public:
        R_OPTIONAL(bool, YaBankCard);
        R_OPTIONAL(bool, YaBankCardOwner);
    };

private:
    R_READONLY(TString, PaymentMethod);
    R_FIELD(TString, Id);
    R_FIELD(TString, Account);
    R_FIELD(TString, CardLevel);
    R_FIELD(TString, CardBank);
    R_FIELD(TString, PSystem);
    R_READONLY(ui32, BIN, 0);
    R_READONLY(TInstant, Binding, TInstant::Zero());
    R_READONLY(TSet<TString>, Aliases);
    R_READONLY(i64, Balance, 0);
    R_READONLY(TString, Suffix);
    R_READONLY(TString, Currency);
    R_READONLY(bool, IsExpired, false);
    R_READONLY(TString, ExpirationYear);
    R_READONLY(TString, ExpirationMonth);
    R_READONLY(TString, Holder);
    R_OPTIONAL(TPayerInfo, PayerInfo);
    R_FIELD(bool, FamilyCardOwner, false);
    R_OPTIONAL(TPartnerInfo, PartnerInfo);

    R_FIELD(TString, Uid);

public:
    TPaymentMethod() = default;

    bool Check(const TString& id) const {
        return Id == id || Aliases.contains(id);
    }

    bool FromJson(const NJson::TJsonValue& json, TMessagesCollector* errors = nullptr);
    NJson::TJsonValue ToJson() const;

    NJson::TJsonValue GetReport() const;

    bool operator<(const TPaymentMethod& other) const {
        if (GetBinding() == other.GetBinding()) {
            return Id < other.GetId();
        }
        return GetBinding() > other.GetBinding();
    }
};

enum class EPaymentStatus {
    NotStarted = 0 /* "not_started" */,
    Started = 1 /* "started" */,
    NotAuthorized = 2 /* "not_authorized" */,
    Authorized = 3 /* "authorized" */,
    Cleared = 4 /* "cleared" */,
    Canceled = 5 /* "canceled" */,
    Refunded = 6 /* "refunded" */,
    Started_3DS = 7 /* "3ds_started" */,
    Unknown = 8 /* "unknown" */,
};

class TPayment {
public:
    using TPtr = TAtomicSharedPtr<TPayment>;

    struct TPayload {
        ui32 ServiceId = 0;
        ui32 BaseAmount = 0;
        TString CashbackType;
        TMaybe<bool> IsPlusUser;
        TString OfferType;
        TString OfferName;
        TString OrderId;
        TString PayloadType;
        TMaybe<NJson::TJsonValue> PayloadExtra;

    public:
        TPayload() = default;
        NJson::TJsonValue ToJson() const;
    };

    class TTerminalRouteData {
        TString Description;
    public:
        TTerminalRouteData(const TString& terminal);
        NJson::TJsonValue ToJson() const;

        explicit operator bool() const {
            return !!Description;
        }
    };

public:
    TPayment() = default;

    void FromJson(const NJson::TJsonValue& json);
    void ToJson(NJson::TJsonValue& json) const;

public:
    NDrive::NTrustClient::EPaymentStatus PaymentStatus = NDrive::NTrustClient::EPaymentStatus::Unknown;
    TString PaymentError;
    TString Status;
    TString PaymentErrorDescription;
    TString PurchaseToken;
    TString Uid;
    TString OrderId;
    TString ProductId;
    TString PaymethodId;
    TString Currency;
    TString Email;
    TString FiscalTitle;
    TString FiscalNDS;
    TString CardMask;
    TString RRN;
    ui32 Amount = 0;
    ui32 Refunded = 0;
    ui32 Cleared = 0;
    TMaybe<TPayload> Payload;
    TMaybe<TTerminalRouteData> TerminalRouteData;
    TMaybe<NJson::TJsonValue> ProcessingInfo;
};

struct TRefund {
    using TPtr = TAtomicSharedPtr<TRefund>;

    TRefund() = default;

    void FromJson(const NJson::TJsonValue& json) {
        RefundStatus = json["payment_status"].GetString();
        Amount = 100.0 * json["amount"].GetDouble();
    }

    void ToJson(NJson::TJsonValue& json) const {
        json["purchase_token"] = PaymentId;
        json["reason_desc"] = "Cancel payment";
        NJson::TJsonValue details;
        details["delta_amount"] = Amount * 0.01;
        details["order_id"] = OrderId;
        json["orders"].AppendValue(details);
    }

public:
    TString PaymentId;
    TString RefundStatus;
    ui64 OrderId;
    ui32 Amount;
};

struct TRefundStatus {
    TString Code;
    TString Description;
};

}}

using TPaymentMethod = NDrive::NTrustClient::TPaymentMethod;
