#include "lpm.h"

#include <library/cpp/threading/future/core/future.h>
#include <rtline/library/unistat/cache.h>
#include <util/stream/file.h>

void TLPMClientConfig::Init(const TYandexConfig::Section* section) {
    auto cacheAdder = [this] (TString type, const TYandexConfig::Section* secPtr) {
        if (type) {
            THolder<NDrive::ITrustStorageConfig> storage(NDrive::ITrustStorageConfig::TFactory::Construct(type, type));
            AssertCorrectConfig(!!storage, "incorrect trust storage type");
            storage->Init(secPtr);
            CacheStorageConfigs.emplace_back(std::move(storage));
        }
    };

    ServiceToken = section->GetDirectives().Value("ServiceToken", ServiceToken);
    ServiceTokenPath = section->GetDirectives().Value("ServiceTokenPath", ServiceTokenPath);
    if (!ServiceToken && ServiceTokenPath) {
        AssertCorrectConfig(TFsPath(ServiceTokenPath).Exists(), "Incorrect path for trust's apikey in 'ServiceTokenPath'");
        TFileInput fi(ServiceTokenPath);
        ServiceToken = fi.ReadAll();
    }
    AssertCorrectConfig(!!ServiceToken, "Incorrect ServiceToken");

    ClientConfig.Init(section);

    /* Support legacy code */
    auto type = section->GetDirectives().Value("CacheType", TString());
    cacheAdder(type, section);
    /* End of legacy block */

    auto sectionChildren = section->GetAllChildren();
    if (auto cacheSec = sectionChildren.find("CacheStorages"); cacheSec != sectionChildren.end()) {
        sectionChildren = cacheSec->second->GetAllChildren();
        for (auto&[secName, secPtr]: sectionChildren) {
            type = secPtr->GetDirectives().Value("CacheType", TString());
            cacheAdder(type, secPtr);
        }
    }
}

THolder<TLPMClient> TLPMClientConfig::Construct(TVector<THolder<NDrive::ITrustStorageOptions>>&& ops, TAtomicSharedPtr<NTvmAuth::TTvmClient> tvmClient) const {
    TAtomicSharedPtr<TTrustClient> client = MakeAtomicShared<TTrustClient>(GetClientConfig(), tvmClient);
    TAtomicSharedPtr<NDrive::ITrustUpdater> updater = MakeAtomicShared<TLPMClientUpdater>(client, ServiceToken);

    return Construct(std::move(ops), updater);
}

THolder<TLPMClient> TLPMClientConfig::Construct(TVector<THolder<NDrive::ITrustStorageOptions>>&& ops, TAtomicSharedPtr<NDrive::ITrustUpdater> updater) const {
    if (ops.size() != CacheStorageConfigs.size()) {
        ERROR_LOG << "Different number of cache options against cache configs";
        return nullptr;
    }

    TVector<std::pair<TString, THolder<NDrive::ITrustStorage>>> cacheStorages;
    for (size_t i = 0; i < ops.size(); i++) {
        THolder<NDrive::ITrustStorage> storage = CacheStorageConfigs[i]->Construct(std::move(ops[i]), MakeHolder<TLPMUpdater>(updater));
        cacheStorages.emplace_back(CacheStorageConfigs[i].Get()->GetCacheId(), std::move(storage));
    }

    return MakeHolder<TLPMClient>(*this, updater, std::move(cacheStorages));
}

void TLPMClientConfig::ToString(IOutputStream& os) const {
    ClientConfig.ToString(os);
    if (ServiceTokenPath) {
        os << "ServiceTokenPath: " << ServiceTokenPath << Endl;
    }

    for (const auto& storage: CacheStorageConfigs) {
        storage->ToString(os);
    }
}

NThreading::TFuture<TVector<NDrive::NTrustClient::TPaymentMethod>> TLPMClientUpdater::GetPaymentMethods(const TString &uid) const {
    if (!Client) {
        return NThreading::MakeErrorFuture<TVector<NDrive::NTrustClient::TPaymentMethod>>(std::make_exception_ptr(yexception() << "incorrect client"));
    }
    return Client->GetPaymentMethods(ServiceToken, uid);
}

NThreading::TFuture<TVector<NDrive::NTrustClient::TPaymentMethod>> TLPMUpdater::GetPaymentMethods(const TString& uid) const {
    if (!Impl) {
        return NThreading::MakeErrorFuture<TVector<NDrive::NTrustClient::TPaymentMethod>>(std::make_exception_ptr(yexception() << "incorrect client"));
    }
    return Impl->GetPaymentMethods(uid);
}

NThreading::TFuture<TVector<NDrive::NTrustClient::TPaymentMethod>> TLPMClient::GetPaymentMethods(const TString& uid, const TString cacheId /* = "" */) const {
    if (!uid) {
        WARNING_LOG << "Got empty uid" << Endl;
        return NThreading::MakeFuture(TVector< NDrive::NTrustClient::TPaymentMethod>());
    }
    if (!cacheId) {
        if (!Updater) {
            return NThreading::MakeErrorFuture<TVector<NDrive::NTrustClient::TPaymentMethod>>(std::make_exception_ptr(yexception() << "incorrect client"));
        }

        TUnistatSignalsCache::SignalAdd("lpm_cache_usage", "default", 1);
        return Updater->GetPaymentMethods(uid);
    }

    if (auto cacheStorage = CacheStorages.find(cacheId); cacheStorage == CacheStorages.end() || !cacheStorage->second.Get()) {
        ERROR_LOG << "Could not find " << cacheId << " in LPM client caches. Redirect query to basic updater" << Endl;
        if (!Updater) {
            return NThreading::MakeErrorFuture<TVector<NDrive::NTrustClient::TPaymentMethod>>(std::make_exception_ptr(yexception() << "incorrect client"));
        }

        TUnistatSignalsCache::SignalAdd("lpm_cache_usage", "default", 1);
        return Updater->GetPaymentMethods(uid);
    } else {
        TUnistatSignalsCache::SignalAdd("lpm_cache_usage", cacheId, 1);
        return cacheStorage->second->GetValue(uid);
    }
}

void TLPMClient::UpdateValue(const TString& uid, const TString& cacheId) const {
    if (auto cacheStorage = CacheStorages.FindPtr(cacheId); cacheStorage && *cacheStorage) {
        (*cacheStorage)->UpdateValue(uid);
    }
}
