#pragma once

#include "client.h"
#include "trust_cache.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/billing/redis_cache.h>
#include <drive/library/cpp/trust/entity.h>

#include <util/generic/ptr.h>

class TLPMClientConfig {
    R_READONLY(TString, ServiceToken);
    R_READONLY(TString, ServiceTokenPath);
    R_READONLY(TTrustClientConfig, ClientConfig);
    R_READONLY(TVector<THolder<NDrive::ITrustStorageConfig>>, CacheStorageConfigs);

public:
    void Init(const TYandexConfig::Section* section);
    void ToString(IOutputStream& os) const;

    static TLPMClientConfig ParseFromString(const TString& configStr) {
        TLPMClientConfig result;
        TAnyYandexConfig config;
        CHECK_WITH_LOG(config.ParseMemory(configStr.data()));
        result.Init(config.GetRootSection());
        return result;
    }

    THolder<TLPMClient> Construct(TVector<THolder<NDrive::ITrustStorageOptions>>&& ops, TAtomicSharedPtr<NTvmAuth::TTvmClient> tvmClient) const;
    THolder<TLPMClient> Construct(TVector<THolder<NDrive::ITrustStorageOptions>>&& ops, TAtomicSharedPtr<NDrive::ITrustUpdater> updater) const;
};

class TLPMClientUpdater : public NDrive::ITrustUpdater {
public:
    TLPMClientUpdater(TAtomicSharedPtr<TTrustClient> client, const TString& serviceToken)
        : Client(client)
        , ServiceToken(serviceToken)
    {
    }

    NThreading::TFuture<TVector<NDrive::NTrustClient::TPaymentMethod>> GetPaymentMethods(const TString& uid) const override;

private:
    TAtomicSharedPtr<TTrustClient> Client;
    TString ServiceToken;
};

class TLPMUpdater : public NDrive::ITrustUpdater {
public:
    TLPMUpdater(TAtomicSharedPtr<NDrive::ITrustUpdater> impl)
        : Impl(impl)
    {
    }

    NThreading::TFuture<TVector<NDrive::NTrustClient::TPaymentMethod>> GetPaymentMethods(const TString& uid) const override;

private:
    TAtomicSharedPtr<NDrive::ITrustUpdater> Impl;
};

class TLPMClient {
public:
    TLPMClient(const TLPMClientConfig& /*config*/, TAtomicSharedPtr<NDrive::ITrustUpdater> updater, TVector<std::pair<TString, THolder<NDrive::ITrustStorage>>>&& cacheStorages)
        : Updater(updater)
    {
        for (auto&& [storageName, storage]: cacheStorages) {
            bool success = CacheStorages.insert({storageName, std::move(storage)}).second;
            if (!success) {
                ERROR_LOG << "Duplicate cacheId in LPM client: " << storageName << Endl;
            }
        }
    }

    NThreading::TFuture<TVector<NDrive::NTrustClient::TPaymentMethod>> GetPaymentMethods(const TString& uid, const TString cacheId = "") const;
    void UpdateValue(const TString& uid, const TString& cacheId) const;

private:
    TAtomicSharedPtr<NDrive::ITrustUpdater> Updater;
    TMap<TString, THolder<NDrive::ITrustStorage>> CacheStorages;
};
