#include "trust_cache.h"

#include <drive/backend/logging/events.h>
#include <rtline/library/json/builder.h>
#include <rtline/library/json/parse.h>
#include <rtline/library/unistat/cache.h>
#include <rtline/util/instant_model.h>

void NDrive::ITrustStorageConfig::Init(const TYandexConfig::Section* section) {
    RefreshInterval = section->GetDirectives().Value("RefreshInterval", RefreshInterval);
    MaxRequests = section->GetDirectives().Value("MaxRequests", MaxRequests);
    SizeLimit = section->GetDirectives().Value("SizeLimit", SizeLimit);
    CacheType = section->GetDirectives().Value("CacheType", CacheType);
    CacheId = section->GetDirectives().Value("CacheId", CacheId);
}

void NDrive::ITrustStorageConfig::ToString(IOutputStream& os) const {
    os << "RefreshInterval: " << RefreshInterval << Endl;
    os << "MaxRequests: " << MaxRequests << Endl;
    os << "SizeLimit: " << SizeLimit << Endl;
    os << "CacheType: " << CacheType << Endl;
    os << "CacheId: " << CacheId << Endl;
}

class TRefreshPayments : public IObjectInQueue {
public:
    TRefreshPayments(const TString& userId, NDrive::ITrustStorage& owner)
        : UserId(userId)
        , Owner(owner)
    {
        Owner.GetGuard().RegisterObject();
    }

    void Process(void* /*threadSpecificResource*/) override {
        Owner.UpdateValue(UserId);
        Owner.GetGuard().UnRegisterObject();
    }

private:
    const TString UserId;
    NDrive::ITrustStorage& Owner;
};

NJson::TJsonValue NDrive::ITrustStorage::TTimedMethods::ToJson() const {
    NJson::TJsonValue result;
    auto& methodsJson = result.InsertValue("methods", NJson::JSON_ARRAY);
    for (const auto& method : Methods) {
        methodsJson.AppendValue(method.ToJson());
    }
    result.InsertValue("deadline", Deadline.Seconds());
    return result;
}

bool NDrive::ITrustStorage::TTimedMethods::FromJson(const NJson::TJsonValue& json) {
    if (!json.Has("methods") || !json["methods"].IsArray()) {
        return false;
    }
    for (const auto& method : json["methods"].GetArray()) {
        NDrive::NTrustClient::TPaymentMethod paymentMethod;
        if (!paymentMethod.FromJson(method)) {
            return false;
        }
        Methods.emplace_back(std::move(paymentMethod));
    }
    return NJson::ParseField(json, "deadline", Deadline);
}

NDrive::ITrustStorage::ITrustStorage(const NDrive::ITrustStorageConfig& config, THolder<NDrive::ITrustUpdater>&& updater)
    : RefreshInterval(config.GetRefreshInterval())
{
    UpdateTasks.Start(config.GetMaxRequests(), config.GetSizeLimit());
    PaymentUpdater = std::move(updater);
}

NDrive::ITrustStorage::~ITrustStorage() {
    UpdateTasks.Stop();
    UpdateGuard.WaitAllTasks();
};

NThreading::TFuture<TVector<NDrive::NTrustClient::TPaymentMethod>> NDrive::ITrustStorage::GetValue(const TString& userId) {
    auto getTimedPayments = GetTimedPayments(userId).Apply([this, userId](const NThreading::TFuture<TTimedMethods>& r) -> NThreading::TFuture<TTimedMethods> {
        if (r.HasValue()) {
            return NThreading::MakeFuture(r.GetValue());
        }

        if (!PaymentUpdater) {
            ythrow yexception() << "incorrect updater";
        }
        return PaymentUpdater->GetPaymentMethods(userId).Apply([](const NThreading::TFuture<TVector<NDrive::NTrustClient::TPaymentMethod>>& r) -> NThreading::TFuture<TTimedMethods> {
            return NThreading::MakeFuture<TTimedMethods>({ TInstant::Zero(), r.GetValue() });
        });
    });
    getTimedPayments.Subscribe([this, userId](const NThreading::TFuture<TTimedMethods>& r) {
        if (!r.HasValue() || ModelingNow() > r.GetValue().Deadline) {
            auto refreshPayments = MakeHolder<TRefreshPayments>(userId, *this);
            if (!UpdateTasks.AddAndOwn(std::move(refreshPayments))) {
                NDrive::TEventLog::Log("RefreshPaymentsAddError", NJson::TMapBuilder
                    ("user_id", userId)
                );
            }
        }
    });
    return getTimedPayments.Apply([](const NThreading::TFuture<TTimedMethods>& r) {
        return r.GetValue().Methods;
    });
}

void NDrive::ITrustStorage::UpdateValue(const TString& userId) {
    if (!PaymentUpdater) {
        return;
    }
    auto payments = PaymentUpdater->GetPaymentMethods(userId);
    payments.Wait();
    if (!payments.HasValue()) {
        return;
    }

    TTimedMethods timedMethods = { ModelingNow() + RefreshInterval, payments.GetValue() };
    DoUpdateValue(userId, timedMethods);
}

void NDrive::ITrustStorage::SignalUpdateResult(const TString& storageId, bool success, TDuration duration) {
    SignalResult("update", success, storageId, duration);
}

void NDrive::ITrustStorage::SignalGetResult(const TString& storageId, bool success, TDuration duration) {
    SignalResult("get", success, storageId, duration);
}

void NDrive::ITrustStorage::SignalResult(const TString& name, bool success, const TString& storageId, TDuration duration) {
    const TString label = success ? "ok" : "failed";
    TUnistatSignalsCache::SignalAdd(name + "-payment-methods-" + storageId, label, 1);
    TUnistatSignalsCache::SignalHistogram(name + "-payment-methods-" + storageId, label + "-times", duration.MilliSeconds(), NRTLineHistogramSignals::IntervalsRTLineReply);
}

NDrive::TLocalTrustStorageConfig::TFactory::TRegistrator<NDrive::TLocalTrustStorageConfig> NDrive::TLocalTrustStorageConfig::Registrator("local");

THolder<NDrive::ITrustStorage> NDrive::TLocalTrustStorageConfig::Construct(THolder<NDrive::ITrustStorageOptions>&& /*ops*/, THolder<NDrive::ITrustUpdater>&& updater) const {
    return MakeHolder<NDrive::TLocalTrustStorage>(*this, std::move(updater));
}

NThreading::TFuture<NDrive::ITrustStorage::TTimedMethods> NDrive::TLocalTrustStorage::GetTimedPayments(const TString& userId) const {
    auto it = TimedMethods.find(userId);
    if (it == TimedMethods.end()) {
        return NThreading::MakeErrorFuture<NDrive::ITrustStorage::TTimedMethods>(std::make_exception_ptr(yexception() << "cannot found " << userId));
    }
    return NThreading::MakeFuture(it->second);
}

void NDrive::TLocalTrustStorage::DoUpdateValue(const TString& userId, const TTimedMethods& payments) {
    TimedMethods[userId] = payments;
}
