#pragma once

#include "entity.h"

#include <drive/library/cpp/wait_guard/wait_guard.h>
#include <rtline/util/queue.h>
#include <library/cpp/threading/future/core/future.h>
#include <library/cpp/object_factory/object_factory.h>
#include <library/cpp/yconf/conf.h>

namespace NDrive {

class ITrustUpdater {
public:
    virtual NThreading::TFuture<TVector<NDrive::NTrustClient::TPaymentMethod>> GetPaymentMethods(const TString& userId) const = 0;
    virtual ~ITrustUpdater() {}
};

class ITrustStorage;
class ITrustStorageOptions;

class ITrustStorageConfig {
    R_READONLY(TDuration, RefreshInterval, TDuration::Minutes(5));
    R_READONLY(ui64, MaxRequests, 8);
    R_READONLY(ui64, SizeLimit, 1000);
    R_READONLY(TString, CacheType, "fake");
    R_READONLY(TString, CacheId, "legacy");

public:
    ITrustStorageConfig(const TString& cacheType = "")
        : CacheType(cacheType)
    {}
    virtual ~ITrustStorageConfig() {}
    virtual void Init(const TYandexConfig::Section* section);
    virtual void ToString(IOutputStream& os) const;

    virtual THolder<ITrustStorage> Construct(THolder<ITrustStorageOptions>&& ops, THolder<ITrustUpdater>&& updater) const = 0;

    using TFactory = NObjectFactory::TParametrizedObjectFactory<ITrustStorageConfig, TString, TString>;
};

class ITrustStorageOptions {
public:
    virtual ~ITrustStorageOptions() = default;

    template <typename T>
    const T* As() const {
        return dynamic_cast<T*>(this);
    }

    template <typename T>
    T* As() {
        return dynamic_cast<T*>(this);
    }
};

class ITrustStorage {
public:
    struct TTimedMethods {
        TInstant Deadline;
        TVector<NDrive::NTrustClient::TPaymentMethod> Methods;

        NJson::TJsonValue ToJson() const;
        bool FromJson(const NJson::TJsonValue& json);
    };

public:
    ITrustStorage(const ITrustStorageConfig& config, THolder<ITrustUpdater>&& updater);

    virtual ~ITrustStorage();

    NThreading::TFuture<TVector<NDrive::NTrustClient::TPaymentMethod>> GetValue(const TString& userId);
    virtual void UpdateValue(const TString& userId);

    TWaitGuard& GetGuard() {
        return UpdateGuard;
    }

protected:
    static void SignalUpdateResult(const TString& storageId, bool success, TDuration duration);
    static void SignalGetResult(const TString& storageId, bool success, TDuration duration);

private:
    virtual NThreading::TFuture<TTimedMethods> GetTimedPayments(const TString& userId) const = 0;
    virtual void DoUpdateValue(const TString& userId, const TTimedMethods& payments) = 0;

    static void SignalResult(const TString& name, bool success, const TString& storageId, TDuration duration);

private:
    TWaitGuard UpdateGuard;
    TCountedMtpQueue UpdateTasks;
    const TDuration RefreshInterval;
    THolder<ITrustUpdater> PaymentUpdater;
};

class TLocalTrustStorageConfig : public ITrustStorageConfig {
public:
    using ITrustStorageConfig::ITrustStorageConfig;
    virtual THolder<ITrustStorage> Construct(THolder<ITrustStorageOptions>&& ops, THolder<ITrustUpdater>&& updater) const override;

private:
    static TFactory::TRegistrator<TLocalTrustStorageConfig> Registrator;
};

class TLocalTrustStorageOptions : public ITrustStorageOptions {
public:
    // Empty
};

class TLocalTrustStorage : public ITrustStorage {
public:
    TLocalTrustStorage(const TLocalTrustStorageConfig& config, THolder<ITrustUpdater>&& updater)
        : ITrustStorage(config, std::move(updater))
    {
    }

private:
    virtual NThreading::TFuture<TTimedMethods> GetTimedPayments(const TString& userId) const override;
    virtual void DoUpdateValue(const TString& userId, const TTimedMethods& payments) override;

private:
    TMap<TString, TTimedMethods> TimedMethods;
};
}

using ITrustStorage = NDrive::ITrustStorage;
