#include "client.h"

#include <drive/library/cpp/geofeatures/client.h>
#include <drive/library/cpp/threading/future.h>

#include <drive/backend/offers/factors/factors_info.h>

#include <library/cpp/mediator/global_notifications/system_status.h>

#include <rtline/api/search_client/neh_client.h>
#include <rtline/api/search_client/reply.h>
#include <rtline/protos/proto_helper.h>

#include <util/generic/algorithm.h>
#include <util/string/builder.h>

namespace {
    using TUserRequestData = NDrive::TUserRequestData;

    NDrive::TRealtimeUserData FillRealtimeUserData(const NRTLine::TSearchReply& reply) {
        NDrive::TRealtimeUserData data;
        auto parser = [&data](const NMetaProtocol::TDocument& document) {
            try {
                if (document.GetUrl().EndsWith(":location")) {
                    data.FillFromLocationDocument(document);
                } else if (document.GetUrl().EndsWith(":data")) {
                    data.FillFromDataDocument(document);
                } else {
                    ERROR_LOG << "Unknown document type " << document.GetUrl() << Endl;
                }
            } catch (const std::exception& e) {
                ERROR_LOG << "Cannot parse events log document " << document.DebugString() << ": "
                          << FormatExc(e) << Endl;
            }
        };
        Y_ENSURE(reply.IsSucceeded(), "Realtime user data fetch has been unsuccessful: "
                                              << reply.GetReqId() << ' ' << reply.GetCode());
        reply.ScanDocs(parser);
        return data;
    }

    NDrive::TRequestHistoryReply FillRequestsHistory(const NRTLine::TSearchReply& reply, ui32 maxRequests) {
        Y_ENSURE(reply.IsSucceeded(), "User requests history fetch has been unsuccessful: "
                                              << reply.GetReqId() << ' ' << reply.GetCode());
        NDrive::TRequestHistoryReply res;
        THashMap<TString, size_t> requestsMap;
        ui32 docsCount = 0;
        auto parser = [&res, &requestsMap, &docsCount, maxRequests](const NMetaProtocol::TDocument& document) {
            ++docsCount;
            try {
                const TString& url = document.GetArchiveInfo().GetUrl();
                TVector<TString> splitted = StringSplitter(url).Split(':').Limit(2);
                Y_ENSURE(splitted.size() == 2, "Unknown document url " << url);
                const TString& event = splitted[0];
                const TString& requestId = splitted[1];

                const auto reqIt = requestsMap.find(requestId);
                size_t currentRequestIndex = 0;
                if (reqIt == requestsMap.end()) {
                    if (res.GetRequests().size() >= maxRequests) {
                        return;
                    }
                    currentRequestIndex = res.MutableRequests().size();
                    res.MutableRequests().push_back(TUserRequestData(requestId));
                    requestsMap[requestId] = currentRequestIndex;
                } else {
                    currentRequestIndex = reqIt->second;
                }

                TUserRequestData& request = res.MutableRequests()[currentRequestIndex];
                if (TUserRequestData::IsAuthorizationEvent(event)) {
                    request.FillFromAuthorizationDocument(document);
                } else if (TUserRequestData::IsResponseEvent(event)) {
                    request.FillFromResponseDocument(document);
                } else {
                    ERROR_LOG << "Unknown document type " << url << Endl;
                }
            } catch (const std::exception& e) {
                ERROR_LOG << "Cannot parse history events document " << document.DebugString()
                          << ": " << FormatExc(e) << Endl;
            }
        };
        reply.ScanDocs(parser);
        res.SetComplete(docsCount < 2*maxRequests);
        return res;
    }

    TString AdoptFactorName(const TString& saasName) {
        bool nextUpper = true;
        TString result;
        for (auto&& i : saasName) {
            if (i == '_') {
                nextUpper = true;
            } else if (nextUpper) {
                result += std::toupper(i);
                nextUpper = false;
            } else {
                result += std::tolower(i);
            }
        }
        return result;
    }

    NDrive::TUserFeatures FillUserFeatures(const NRTLine::TSearchReply& reply) {
        Y_ENSURE(reply.IsSucceeded(), reply.GetCode() << ' ' << reply.GetReqId());
        NDrive::TUserFeatures result;
        reply.ScanDocs([&result](const NMetaProtocol::TDocument& document) {
            const IFactorsInfo* floatFactorsInfo = NDrive::GetOfferFactorsInfo();

            const IFactorsInfo* categorialFactorsInfo2 = NDrive::GetCatOfferFactorsInfo2();
            const IFactorsInfo* floatFactorsInfo2 = NDrive::GetOfferFactorsInfo2();
            if (floatFactorsInfo) {
                result.Factors.resize(floatFactorsInfo->GetFactorCount());
            }

            if (categorialFactorsInfo2) {
                result.CatFactors2.resize(categorialFactorsInfo2->GetFactorCount());
            }
            if (floatFactorsInfo2) {
                result.Factors2.resize(floatFactorsInfo2->GetFactorCount());
            }
            for (auto&& i : document.GetArchiveInfo().GetGtaRelatedAttribute()) {
                const TString& key = i.GetKey();
                const TString& value = i.GetValue();
                TString adoptedKey;
                if (categorialFactorsInfo2) {
                    if (!adoptedKey) {
                        adoptedKey = AdoptFactorName(key);
                    }
                    {
                        auto index = categorialFactorsInfo2->GetFactorIndex(adoptedKey.c_str());
                        if (index && index < result.CatFactors2.size()) {
                            result.CatFactors2[*index] = value;
                        }
                    }
                }
                if (value != "null") {
                    if (!adoptedKey) {
                        adoptedKey = AdoptFactorName(key);
                    }
                    if (floatFactorsInfo) {
                        auto index = floatFactorsInfo->GetFactorIndex(adoptedKey.c_str());
                        if (index && *index < result.Factors.size()) {
                            result.Factors[*index] = FromString(value);
                        }
                    }
                    if (floatFactorsInfo2) {
                        auto index = floatFactorsInfo2->GetFactorIndex(adoptedKey.c_str());
                        if (index && *index < result.Factors.size()) {
                            result.Factors2[*index] = FromString(value);
                        }
                    }
                }
                if (key == "average_session_cars_count") {
                    result.AverageSessionCarsCount = FromString(value);
                    continue;
                }
                if (key == "average_session_clicks_count") {
                    result.AverageSessionClicksCount = FromString(value);
                    continue;
                }
                if (key == "installed_apps") {
                    result.InstalledApps.insert(value);
                    continue;
                }
            }
            result.Defined = true;
        });
        return result;
    }

    NDrive::TOptionalUserGeoFeatures FillUserGeoFeatures(const NRTLine::TSearchReply& reply) {
        Y_ENSURE(reply.IsSucceeded(), reply.GetCode() << ' ' << reply.GetReqId());
        bool defined = false;
        NDrive::TUserGeoFeatures result;
        reply.ScanDocs([&defined, &result](const NMetaProtocol::TDocument& document) {
            TReadSearchProtoHelper helper(document);
            auto fields = result.GetFields();
            ForEach(fields, [&defined, &helper] (auto&& field) {
                defined |= helper.GetProperty(field.GetName(), field.Value);
            });
        });
        if (!defined) {
            return {};
        }
        return result;
    }

    NDrive::TOptionalUserDoubleGeoFeatures FillUserDoubleGeoFeatures(const NRTLine::TSearchReply& reply) {
        Y_ENSURE(reply.IsSucceeded(), reply.GetCode() << ' ' << reply.GetReqId());
        bool defined = false;
        NDrive::TUserDoubleGeoFeatures result;
        reply.ScanDocs([&defined, &result](const NMetaProtocol::TDocument& document) {
            TReadSearchProtoHelper helper(document);
            auto fields = result.GetFields();
            ForEach(fields, [&defined, &helper] (auto&& field) {
                defined |= helper.GetProperty(field.GetName(), field.Value);
            });
        });
        if (!defined) {
            return {};
        }
        return result;
    }

    NDrive::TUserDestinationSuggest FillUserDestinationSuggest(const NRTLine::TSearchReply& reply) {
        Y_ENSURE(reply.IsSucceeded(), reply.GetCode() << ' ' << reply.GetReqId());
        NDrive::TUserDestinationSuggest result;
        reply.ScanDocs([&result](const NMetaProtocol::TDocument& document) {
            NDrive::TUserDestinationSuggest::TElement element;
            for (auto&& i : document.GetArchiveInfo().GetGtaRelatedAttribute()) {
                const TString& key = i.GetKey();
                const TString& value = i.GetValue();
                if (key == "latitude") {
                    element.Latitude = FromString(value);
                    continue;
                }
                if (key == "longitude") {
                    element.Longitude = FromString(value);
                    continue;
                }
            }
            if (element) {
                result.Elements.push_back(std::move(element));
            }
        });
        return result;
    }
}  // namespace

namespace NDrive {
    void TUserEventsApiConfig::Init(const TYandexConfig::Section* section) {
        KVRTLineApiName = section->GetDirectives().Value<TString>("RTLineApiName", KVRTLineApiName);
        AssertCorrectConfig(!!KVRTLineApiName, "Missing KVRTLineApi for UserEventsApi.");
        KVResponseTimeout = section->GetDirectives().Value<TDuration>("ResponseTimeout", KVResponseTimeout);
        HistoryRTLineApiName = section->GetDirectives().Value<TString>("HistoryRTLineApiName", HistoryRTLineApiName);
        AssertCorrectConfig(!!HistoryRTLineApiName, "Missing HistoryRTLineApi for UserEventsApi.");
        HistoryResponseTimeout = section->GetDirectives().Value<TDuration>("HistoryResponseTimeout", HistoryResponseTimeout);
        MaxHistoryRequestsCount = section->GetDirectives().Value<ui32>("MaxHistoryRequestsCount", MaxHistoryRequestsCount);
    }

    void TUserEventsApiConfig::ToString(IOutputStream& os) const {
        os << "KVRTLineApiName: " << KVRTLineApiName << Endl;
        os << "HistoryRTLineApiName: " << HistoryRTLineApiName << Endl;
        os << "KVResponseTimeout: " << KVResponseTimeout << Endl;
        os << "HistoryResponseTimeout: " << HistoryResponseTimeout << Endl;
        os << "MaxHistoryRequestsCount: " << MaxHistoryRequestsCount << Endl;
    }

    TUserEventsApi::TUserEventsApi(const TUserEventsApiConfig& config, const NRTLine::TNehSearchClient& kvClient, const NRTLine::TNehSearchClient& historyClient)
        : KVClient(kvClient)
        , HistoryClient(historyClient)
        , KVResponseTimeout(config.GetKVResponseTimeout())
        , HistoryResponseTimeout(config.GetHistoryResponseTimeout())
        , MaxHistoryRequestsCount(config.GetMaxHistoryRequestsCount())
    {
    }

    NThreading::TFuture<TRealtimeUserData> TUserEventsApi::GetRealtimeUserInfo(const TString& userId, bool fetchLocation, bool fetchClientData) const {
        if (!fetchLocation && !fetchClientData) {
            return NThreading::MakeFuture<TRealtimeUserData>();
        }

        auto query = NRTLine::TQuery();
        TStringBuilder keys;
        if (fetchLocation) {
            keys << "user_id:" << userId << ":location";
        }
        if (fetchClientData) {
            if (!keys.empty()) {
                keys << ",";
            }
            keys << "user_id:" << userId << ":data";
        }
        query.SetText(keys);
        NThreading::TFuture<NRTLine::TSearchReply> reply = KVClient.SendAsyncQueryF(query, KVResponseTimeout);
        return reply.Apply([](const NThreading::TFuture<NRTLine::TSearchReply>& r) {
            return FillRealtimeUserData(r.GetValue());
        });
    }

    NThreading::TFuture<TUserFeatures> TUserEventsApi::GetUserFeatures(const TString& userId) const {
        auto query = NRTLine::TQuery();
        query.SetText(TStringBuilder() << "user_id:" << userId << ":features");
        NThreading::TFuture<NRTLine::TSearchReply> reply = KVClient.SendAsyncQueryF(query, KVResponseTimeout);
        NThreading::TFuture<TUserFeatures> result = reply.Apply([](const NThreading::TFuture<NRTLine::TSearchReply>& r) {
            return FillUserFeatures(r.GetValue());
        });
        return result;
    }

    NThreading::TFuture<TOptionalUserGeoFeatures> TUserEventsApi::GetUserGeoFeatures(const TString& userId, const TGeoCoord& coordinate) const {
        auto latitude = coordinate.Y;
        auto longitude = coordinate.X;
        auto query = NRTLine::TQuery();
        query.SetText(TStringBuilder() << "user_id:" << userId << ":" << TGeoFeaturesClient::GetUrl(latitude, longitude));
        auto reply = KVClient.SendAsyncQueryF(query, KVResponseTimeout);
        return reply.Apply([](const NThreading::TFuture<NRTLine::TSearchReply>& r) {
            return FillUserGeoFeatures(r.GetValue());
        });
    }

    NThreading::TFuture<TOptionalUserGeoFeatures> TUserEventsApi::GetUserGeoFeatures(const TString& userId, ui64 geobaseId) const {
        auto query = NRTLine::TQuery();
        query.SetText(TStringBuilder() << "user_id:" << userId << ":" << TGeoFeaturesClient::GetUrl(geobaseId));
        auto reply = KVClient.SendAsyncQueryF(query, KVResponseTimeout);
        return reply.Apply([](const NThreading::TFuture<NRTLine::TSearchReply>& r) {
            return FillUserGeoFeatures(r.GetValue());
        });
    }

    NThreading::TFuture<TOptionalUserDoubleGeoFeatures> TUserEventsApi::GetUserDoubleGeoFeatures(const TString& userId, const TGeoCoord& a, const TGeoCoord& b) const {
        auto query = NRTLine::TQuery();
        query.SetText(TStringBuilder()
            << "user_id:" << userId << ":"
            << "geo_id_a:" << TGeoFeaturesClient::GetGeoId(a.Y, a.X) << ":"
            << "geo_id_b:" << TGeoFeaturesClient::GetGeoId(b.Y, b.X)
        );
        auto reply = KVClient.SendAsyncQueryF(query, KVResponseTimeout);
        return reply.Apply([](const NThreading::TFuture<NRTLine::TSearchReply>& r) {
            return FillUserDoubleGeoFeatures(r.GetValue());
        });
    }

    NThreading::TFuture<TOptionalUserDoubleGeoFeatures> TUserEventsApi::GetUserDoubleGeoFeatures(const TString& userId, ui64 geobaseIdA, ui64 geobaseIdB) const {
        auto query = NRTLine::TQuery();
        query.SetText(TStringBuilder()
            << "user_id:" << userId << ":"
            << "geoid_a:" << geobaseIdA << ":"
            << "geoid_b:" << geobaseIdB << ":"
            << "features"
        );
        auto reply = KVClient.SendAsyncQueryF(query, KVResponseTimeout);
        return reply.Apply([](const NThreading::TFuture<NRTLine::TSearchReply>& r) {
            return FillUserDoubleGeoFeatures(r.GetValue());
        });
    }

    NThreading::TFuture<TRequestHistoryReply> TUserEventsApi::GetUserRequests(const TString& userId, TInstant since, TInstant until, ui32 maxRequestsCount) const {
        auto query = NRTLine::TQuery();
        TStringBuilder queryText;
        queryText << "userid:" << userId;
        if (since != TInstant::Zero()) {
            queryText << " " << "timeround:>=" << (since.Seconds() / 1000) * 1000;
        }
        if (until != TInstant::Max()) {
            queryText << " " << "timeround:<" << until.Seconds();
        }
        query.SetText(queryText);

        const ui32 requestsCount = maxRequestsCount ? maxRequestsCount : MaxHistoryRequestsCount;
        // TotalDocsCount in saas response is not trusted, so we ask for more documents
        // that we really needs to check completeness.
        query.SetNumDoc(2 * requestsCount + 1);

        query.HowBuilder().SetAttrName("Timestamp");
        query.HowBuilder().SetSortMethod(NRTLine::TSortPolicy::smAttr);

        auto reply = HistoryClient.SendAsyncQueryF(query, HistoryResponseTimeout);
        return reply.Apply([requestsCount](const NThreading::TFuture<NRTLine::TSearchReply>& r) {
            return FillRequestsHistory(r.GetValue(), requestsCount);
        });
    }

    NThreading::TFuture<TUserDestinationSuggest> TUserEventsApi::GetUserDestinationSuggest(const TString& userId, bool fetchGeoFeatures) const {
        auto query = NRTLine::TQuery();
        query.AddExtraParam("key_name", "destination_user_id");
        query.SetText(userId);
        query.SetNumDoc(Max<ui16>());
        auto reply = KVClient.SendAsyncQueryF(query, KVResponseTimeout);
        auto suggest = reply.Apply([](const NThreading::TFuture<NRTLine::TSearchReply>& r) {
            return FillUserDestinationSuggest(r.GetValue());
        });
        if (fetchGeoFeatures) {
            auto suggestWithFeatures = suggest.Apply([this, userId](const NThreading::TFuture<TUserDestinationSuggest>& s) {
                TUserDestinationSuggest result = s.GetValue();
                for (auto&& element : result.Elements) {
                    TGeoCoord coordinate = { element.Longitude, element.Latitude };
                    element.UserGeoFeatures = GetUserGeoFeatures(userId, coordinate);
                }
                return result;
            });
            return suggestWithFeatures;
        } else {
            return suggest;
        }
    }
}
