#include "user_requests_history.h"

#include <search/idl/meta.pb.h>

namespace {
    const TString AUTHORIZATION_EVENT = "Authorization";
    const TSet<TString> RESPONSE_EVENTS = {
            "CompressedResponse",
            "ResponseInfo",
            "Response",
            "Timeout",
    };
}  // namespace

namespace NDrive {
    TUserRequestData::TUserRequestData(const TString& requestId)
        : RequestId(requestId)
    {
    }

    bool TUserRequestData::IsAuthorizationEvent(const TString& logEvent) {
        return logEvent == AUTHORIZATION_EVENT;
    }

    bool TUserRequestData::IsResponseEvent(const TString& logEvent) {
        return RESPONSE_EVENTS.contains(logEvent);
    }

    NJson::TJsonValue TUserRequestData::SerializeToJson(bool reportLocation,
                                                        bool reportClientAppData) const {
        NJson::TJsonValue res(NJson::JSON_MAP);
        if (!Initialized()) {
            return res;
        }

        if (reportLocation) {
            res["latitude"] = Location.Y;
            res["longitude"] = Location.X;
        }

        if (reportClientAppData) {
            res["user_agent"] = UserAgent;
            res["ip"] = Ip;
            res["device_id"] = DeviceId;
            res["app_version"] = AppVersion;
        }

        res["reqid"] = RequestId;
        res["timestamp"] = Timestamp.Seconds();
        res["host"] = Host;
        res["path"] = Source;
        res["query"] = Query;
        res["response"] = Response;
        res["post"] = PostData;
        res["code"] = Code;
        TJsonProcessor::WriteContainerArray(res, "experiments", Experiments);
        return res;
    }

    bool TUserRequestData::Initialized() const {
        return Timestamp != TInstant::Zero();
    }

    void TUserRequestData::FillFromAuthorizationDocument(const NMetaProtocol::TDocument& document) {
        for (auto&& property : document.GetArchiveInfo().GetGtaRelatedAttribute()) {
            const auto& key = property.GetKey();
            const auto& value = property.GetValue();
            if (key == "MTime") {
                Timestamp = TInstant::Seconds(FromString<ui64>(value));
            } else if (key == "Latitude") {
                Location.Y = FromString<double>(value);
            } else if (key == "Longitude") {
                Location.X = FromString<double>(value);
            } else if (key == "Ip") {
                Ip = value;
            } else if (key == "Source") {
                Source = value;
            } else if (key == "Host") {
                Host = value;
            } else if (key == "UserAgent") {
                UserAgent = value;
            } else if (key == "AppVersion") {
                AppVersion = value;
            } else if (key == "PostData") {
                PostData = value;
            } else if (key == "DeviceID") {
                DeviceId = value;
            } else if (key == "Experiments") {
                StringSplitter(value).Split(',').ParseInto(&Experiments);
            }
        }
    }

    void TUserRequestData::FillFromResponseDocument(const NMetaProtocol::TDocument& document) {
        for (auto&& property : document.GetArchiveInfo().GetGtaRelatedAttribute()) {
            const auto& key = property.GetKey();
            const auto& value = property.GetValue();
            if (key == "Query") {
                Query = value;
            } else if (key == "Code") {
                Code = FromString(value);
            } else if (key == "Data") {
                Response = value;
            }
        }
    }

    NJson::TJsonValue TRequestHistoryReply::SerializeToJson(
        bool reportLocation,
        bool reportClientAppData,
        TInstant since,
        TInstant until
    ) const {
        NJson::TJsonValue res(NJson::JSON_MAP);
        auto& requestsJson = res.InsertValue("requests", NJson::JSON_ARRAY);
        for (const auto& r : Requests) {
            auto timestamp = r.GetTimestamp();
            if (timestamp < since) {
                continue;
            }
            if (!(timestamp < until)) {
                continue;
            }
            if (r.Initialized()) {
                requestsJson.AppendValue(r.SerializeToJson(reportLocation, reportClientAppData));
            } else {
                ERROR_LOG << "Skipping request " << r.GetRequestId() << " not initialized";
            }
        }
        res["complete"] = Complete;
        return res;
    }
}  // namespace NDrive
