#include "weather_api.h"

#include <drive/library/cpp/weather/proto/weather.pb.h>

#include <drive/backend/abstract/settings.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/mediator/messenger.h>
#include <library/cpp/svnversion/svnversion.h>

#include <rtline/library/unistat/signals.h>

bool TWeatherAPI::DoStart() {
    AD = new TAsyncDelivery();
    AD->Start(Config.GetThreadsMC(), Config.GetThreadsRequest());
    Agent.Reset(new NNeh::THttpClient(AD));
    Agent->RegisterSource("weather", Config.GetHost(), Config.GetPort(), Config.GetReaskConfig(), Config.GetHttps());
    return true;
}

bool TWeatherAPI::DoStop() {
    AD->Stop();
    return true;
}

namespace {
    TNamedSignalCustom WeatherRequestsSuccess("frontend-weather-success", EAggregationType::Sum, "dxxx");
    TNamedSignalCustom WeatherRequestsFail("frontend-weather-fail", EAggregationType::Sum, "dxxx");
    TNamedSignalCustom WeatherRequestsDuration("frontend-weather-duration", EAggregationType::Sum, "dxxx");
}

TMap<TString, TWeatherInfo> TWeatherAPI::GetWeather(const TMap<TString, TGeoCoord>& coords) const {
    CHECK_WITH_LOG(IsActive());
    const TInstant deadline = Now() + Config.GetReaskConfig().GetGlobalTimeout();
    TMap<TString, NNeh::THttpRequest> requests;
    for (auto&& i : coords) {
        NNeh::THttpRequest request;
        request.SetUri(Config.GetUri()).SetCgiData("&extra=1&lat=" + ::ToString(i.second.Y) + "&lon=" + ::ToString(i.second.X));
        request.AddHeader("User-Agent", "drive-backend/" + ::ToString(GetProgramSvnRevision()));
        request.AddHeader("X-Yandex-API-Key", Config.GetAuthToken());
        requests.emplace(i.first, std::move(request));
    }

    const ui32 maxInFly = Config.GetMaxInFly();

    const TInstant start = Now();

    TMap<TString, NUtil::THttpReply> reqResult = Agent->SendPack(requests, deadline, maxInFly);

    const TDuration weatherRequestsDuration = Now() - start;

    TMap<TString, TWeatherInfo> result;

    ui32 countSuccess = 0;
    ui32 countFails = 0;
    for (auto&& i : reqResult) {
        const NUtil::THttpReply& reply = i.second;
        TWeatherInfo wInfo(reply);
        if (wInfo.GetIsCorrect()) {
            ++countSuccess;
        } else {
            ++countFails;
        }
        result.emplace(i.first, std::move(wInfo));
    }
    WeatherRequestsDuration.Signal(weatherRequestsDuration.MilliSeconds());
    WeatherRequestsSuccess.Signal(countSuccess);
    WeatherRequestsFail.Signal(countFails);
    return result;
}

bool TWeatherInfo::ParseReport(const NJson::TJsonValue& report) {
    const NJson::TJsonValue::TMapType* mapCurrent;
    if (!report.GetMapPointer(&mapCurrent)) {
        return false;
    }
    {
        auto it = mapCurrent->find("cloudness");
        if (it == mapCurrent->end() || !it->second.IsDouble()) {
            return false;
        } else {
            Cloudness = it->second.GetDouble();
            if (Cloudness < 0 || Cloudness > 1) {
                return false;
            }
        }
    }
    {
        auto it = mapCurrent->find("prec_strength");
        if (it == mapCurrent->end() || !it->second.IsDouble()) {
            return false;
        } else {
            PrecStrength = it->second.GetDouble();
            if (PrecStrength < 0 || PrecStrength > 1) {
                return false;
            }
        }
    }
    {
        auto it = mapCurrent->find("prec_type");
        if (it == mapCurrent->end() || !it->second.IsUInteger() || it->second.GetUInteger() >= (ui32)EWeatherPrecType::COUNT) {
            return false;
        } else {
            PrecType = (EWeatherPrecType)it->second.GetUInteger();
        }
    }
    return true;
}

TWeatherInfo::TWeatherInfo(const NUtil::THttpReply& report): IsCorrect(report.Code() == 200) {
    if (IsCorrect) {
        NJson::TJsonValue reportJson;
        if (NJson::ReadJsonFastTree(report.Content(), &reportJson)) {
            IsCorrect = ParseReport(reportJson);
        } else {
            IsCorrect = false;
        }

    }
}

NDrive::NProto::TWeather TWeatherInfo::SerializeToProto() const {
    NDrive::NProto::TWeather result;
    result.SetCloudness(static_cast<float>(Cloudness));
    result.SetPrecType(static_cast<ui32>(PrecType));
    result.SetPrecStrength(static_cast<float>(PrecStrength));
    return result;
}

bool TWeatherInfo::DeserializeFromProto(const NDrive::NProto::TWeather& info) {
    PrecStrength = info.GetPrecStrength();
    Cloudness = info.GetCloudness();
    if (info.GetPrecType() >= (ui32)EWeatherPrecType::COUNT) {
        return false;
    }
    PrecType = (EWeatherPrecType)info.GetPrecType();
    return true;
}

NJson::TJsonValue TWeatherInfo::SerializeToJson() const {
    NJson::TJsonValue result;
    JWRITE_DEF(result, "correct", IsCorrect, true);
    JWRITE_DEF(result, "p_s", PrecStrength, 0);
    JWRITE_DEF(result, "cl", Cloudness, 0);
    JWRITE_ENUM(result, "p_t", PrecType);
    return result;
}

bool TWeatherInfo::DeserializeFromJson(const NJson::TJsonValue& info) {
    IsCorrect = true;
    JREAD_DOUBLE_OPT(info, "p_s", PrecStrength);
    JREAD_DOUBLE_OPT(info, "cl", Cloudness);
    JREAD_FROM_STRING(info, "p_t", PrecType);
    return true;
}
