#include "client.h"

#include <rtline/library/json/builder.h>

#include <util/string/subst.h>

TWebphoneClient::TResponse::TResponse(ui32 code, const TString& content)
    : Code(code)
    , Content(content)
{
}

NJson::TJsonValue TWebphoneClient::TResponse::GetJsonContent() const {
    NJson::TJsonValue jsonContent;
    return (NJson::ReadJsonFastTree(Content, &jsonContent)) ? jsonContent : NJson::TJsonValue(Content);
}

NJson::TJsonValue TWebphoneClient::TResponse::SerializeToJson() const {
    return NJson::TMapBuilder("error_code", GetCode())("error_content", GetJsonContent());
}

TWebphoneClient::TWebphoneClient(const TWebphoneClientConfig& config, TAtomicSharedPtr<NTvmAuth::TTvmClient> tvmClient)
    : Config(config)
    , AD(new TAsyncDelivery())
    , Agent(new NNeh::THttpClient(AD))
    , TvmAuth(tvmClient && config.GetDestinationTvmId() ? MakeMaybe<NDrive::TTvmAuth>(tvmClient, config.GetDestinationTvmId()) : Nothing())
{
    AD->Start(Config.GetRequestConfig().GetThreadsStatusChecker(), Config.GetRequestConfig().GetThreadsSenders());
    Agent->RegisterSource("webphone-client", Config.GetHost(), Config.GetPort(), Config.GetRequestConfig(), true);
}

TWebphoneClient::~TWebphoneClient() {
    AD->Stop();
}

NThreading::TFuture<TWebphoneClient::TResponse> TWebphoneClient::GetWebphoneAuthData(const TString& login, const NJson::TJsonValue& payload) const {
    return GetWebphoneAuthData(login, "", payload);
}

NThreading::TFuture<TWebphoneClient::TResponse> TWebphoneClient::GetWebphoneAuthData(const TString& login, const TString& cookie, const NJson::TJsonValue& payload) const {
    TString uri = Config.GetGenerateTokenUri();
    SubstGlobal(uri, "{$login}", login);

    NNeh::THttpRequest request;
    request
        .SetUri(uri)
        .SetPostData(payload.GetStringRobust())
        .SetRequestType("POST")
        .AddHeader("Content-Type", "application/json");

    if (cookie) {
        request.AddHeader("Cookie", cookie);
    } else {
        if (!TvmAuth) {
            return NThreading::MakeErrorFuture<TWebphoneClient::TResponse>(std::make_exception_ptr(yexception() << "TvmClient not configured"));
        }
        TvmAuth->UpdateRequest(request);
    }

    return Agent->SendAsync(request, Now() + Config.GetRequestTimeout()).Apply([](const NThreading::TFuture<NUtil::THttpReply>& r) {
        const auto& reply = r.GetValue();
        return TResponse(reply.Code(), reply.Content());
    });
}
