#include "client.h"

#include <drive/library/cpp/raw_text/datetime.h>
#include <drive/library/cpp/threading/future.h>

#include <rtline/library/json/cast.h>
#include <rtline/library/json/parse.h>
#include <rtline/util/json_processing.h>

#include <util/string/builder.h>
#include <util/string/subst.h>

bool ParseWithTimeZone(const NJson::TJsonValue& jsonDate, TInstant& result) {
    TInstant moscowTime;
    if (!NJson::ParseField(jsonDate, moscowTime)) {
        return false;
    }
    NUtil::TTimeZone moscowTimeZone = NUtil::GetTimeZone("Europe/Moscow");
    result = NUtil::ConvertTimeZone(moscowTime, moscowTimeZone, NUtil::GetUTCTimeZone());
    return true;
}

bool TWhsdTransponderEvent::DeserializeFromJson(const NJson::TJsonValue& jsonEvent, TMessagesCollector& errors) {
    double amountD;
    if (!NJson::ParseField(jsonEvent["pan"], TransponderId, true) || !NJson::ParseField(jsonEvent["amount"], amountD, true)) {
        errors.AddMessage("TWhsdTransponderEvent parsing", "no required fields pan, amount");
        return false;
    }
    Amount = std::round(amountD * 100.0);

    if (!ParseWithTimeZone(jsonEvent["entry_date"], Date) && !ParseWithTimeZone(jsonEvent["dt"], Date)) {
        errors.AddMessage("TWhsdTransponderEvent parsing", "couldn't parse event date");
        return false;
    }
    if (!NJson::ParseField(jsonEvent["rid"], Id)) {
        errors.AddMessage("TWhsdTransponderEvent parsing", "couldn't parse event id");
    }
    if (!NJson::ParseField(jsonEvent["entry_place"], EntryLocation)) {
        errors.AddMessage("TWhsdTransponderEvent parsing", "couldn't parse event entry place");
    }
    if (!NJson::ParseField(jsonEvent["place"], ExitLocation)) {
        errors.AddMessage("TWhsdTransponderEvent parsing", "couldn't parse event exit place");
    }
    return true;
}

TString DateToRequestString(const TInstant date) {
    NUtil::TTimeZone moscowTimeZone = NUtil::GetTimeZone("Europe/Moscow");
    TInstant moscowTime = NUtil::ConvertTimeZone(date, NUtil::GetUTCTimeZone(), moscowTimeZone);
    return moscowTime.FormatGmTime("%Y-%m-%dT%H:%M:%S");
}

NNeh::THttpRequest TWhsdClient::CreateRequest(const TString& from, const TString& to, const ui32 rowsSkip, const ui32 rowsLimit) const {
    TString uri = TStringBuilder()
                  << "/onyma/system/api/json?function=onm_api_toll_api_mobile_wall"
                  << "&auth_token=" << Token
                  << "&rows_limit=" << rowsLimit
                  << "&rows_skip=" << rowsSkip
                  << "&event_type=" << "{\"in\":[1]}"
                  << R"(&dt={"bw":[")" << from << "\",\"" << to <<"\"]}";
    NNeh::THttpRequest request;
    request.SetUri(uri);
    request.SetRequestType("GET");
    return request;
}

NThreading::TFuture<TWhsdRequestResult> TWhsdClient::GetTransponderEvents(const TInstant from, const TInstant to) const {
    return GetTransponderEvents(from, to, 0, false);
}

NThreading::TFuture<TWhsdRequestResult> TWhsdClient::GetTransponderEvents(const TInstant from, const TInstant to, const ui32 rowsToSkip, bool tryReAuth) const {
    if (!Authorize(tryReAuth)) {
        if (!tryReAuth) {
            return GetTransponderEvents(from, to, rowsToSkip, true);
        } else {
            return NThreading::MakeErrorFuture<TWhsdRequestResult>(std::make_exception_ptr(yexception() << "Can't open session"));
        }
    }
    auto request = CreateRequest(DateToRequestString(from), DateToRequestString(to), rowsToSkip, Config.GetRowsToRead());
    return SendRequest(EWhsdOperationType::GetTransponderEvents, request).Apply([this, from, to, rowsToSkip, tryReAuth](const NThreading::TFuture<NJson::TJsonValue>& r)  {
        if (r.HasException()) {
            if (!tryReAuth) {
                return GetTransponderEvents(from, to, rowsToSkip, true);
            } else {
                throw NThreading::GetException(r);
            }
        }
        NJson::TJsonValue replyJson = r.GetValue();
        bool moreRows = false;
        if (!NJson::ParseField(replyJson["more_rows"], moreRows)) {
            LogError(EWhsdOperationType::ParseEvent);
            throw yexception() << "Error parsing more_rows field";
        }
        if (!replyJson["return"].IsArray()) {
            LogError(EWhsdOperationType::ParseEvent);
            throw yexception() << "'return' is not an array";
        }
        TVector<TWhsdTransponderEvent> events;
        for (auto&& jsonEvent : replyJson["return"].GetArray()) {
            TWhsdTransponderEvent event;
            if (event.DeserializeFromJson(jsonEvent, Errors)) {
                events.push_back(std::move(event));
            } else {
                LogError(EWhsdOperationType::ParseEvent);
                throw NThreading::MakeErrorFuture<TWhsdRequestResult>(std::make_exception_ptr(yexception() << "Error parsing events"));
            }
        }
        if (moreRows) {
            return NThreading::MakeFuture(TWhsdRequestResult(events, GetTransponderEvents(from, to, rowsToSkip + Config.GetRowsToRead(), tryReAuth)));
        } else {
            return NThreading::MakeFuture(TWhsdRequestResult(events));
        }
    });
}

NThreading::TFuture<NJson::TJsonValue> TWhsdClient::SendRequest(const EWhsdOperationType operationType, const NNeh::THttpRequest& request) const {
    if (!Agent) {
        return NThreading::MakeErrorFuture<NJson::TJsonValue>(std::make_exception_ptr(yexception() << "HttpClient not configured"));
    }
    return Agent->SendAsync(request, Now() + Config.GetRequestTimeout()).Apply([this, operationType](const NThreading::TFuture<NUtil::THttpReply>& r) {
        if (r.HasException()) {
            throw NThreading::GetException(r);
        }
        const auto& reply = r.GetValue();
        NJson::TJsonValue replyJson;
        Logger.ProcessReply(operationType, reply.Code());
        if (reply.Code() != 200 || !NJson::ReadJsonFastTree(reply.Content(), &replyJson)) {
            throw yexception() << "Send request error, reply code " << reply.Code() << ", " << reply.ErrorMessage();
        }
        return replyJson;
    });
}

bool TWhsdClient::Authorize(const bool refreshToken /*= false*/) const {
    if (!refreshToken && Token) {
        return true;
    }
    if (!Config.GetLogin() || !Config.GetPassword()) {
        LogError(EWhsdOperationType::OpenSession, "Password or login not configured");
        return false;
    }
    TString uri = TStringBuilder()
                  << "/onyma/system/api/json?function=open_session"
                  << "&user=" << Config.GetLogin()
                  << "&pass=" << Config.GetPassword()
                  << "&realm=WHSD";
    NNeh::THttpRequest request;
    request.SetUri(uri).SetRequestType("GET");
    auto authFuture = SendRequest(EWhsdOperationType::OpenSession, request);
    if (authFuture.Wait(Config.GetRequestTimeout()) && authFuture.HasValue()) {
        NJson::TJsonValue replyJson = authFuture.GetValue();
        TString token;
        if (!NJson::ParseField(replyJson["return"], token, true) || !token) {
            LogError(EWhsdOperationType::OpenSession, "Could not parse open_session reply: " + replyJson.GetStringRobust());
            return false;
        }
        Token = token;
        return true;
    } else if (authFuture.HasException()) {
        LogError(EWhsdOperationType::OpenSession, "Network request error: " + NThreading::GetExceptionMessage(authFuture));
        return false;
    } else {
        LogError(EWhsdOperationType::OpenSession, "Network request error: request timeout");
    }
    return false;
}

void TWhsdClient::CloseSession() const {
    if (!Token) {
        return;
    }
    TString uri = TStringBuilder()
                  << "/onyma/system/api/json?function=close_session"
                  << "&auth_token=" << Token;
    NNeh::THttpRequest request;
    request.SetUri(uri).SetRequestType("GET");
    SendRequest(EWhsdOperationType::CloseSession, request).Apply([this](const NThreading::TFuture<NJson::TJsonValue>& r)  {
        if (!r.HasException()) {
            Token = "";
        }
    });
}

void TWhsdClient::LogError(const EWhsdOperationType& type, const TString& message /*= ""*/) const {
    if (message) {
        Logger.ProcessError(type, message, Errors);
    } else {
        Logger.ProcessError(type);
    }
}

void TTestWhsdClient::LogError(const EWhsdOperationType& type, const TString& message /*= ""*/) const {
    Errors.AddMessage("TestWhsdClient " + ToString(type), message);
}

NThreading::TFuture<NJson::TJsonValue> TTestWhsdClient::SendRequest(EWhsdOperationType operationType, const NNeh::THttpRequest& request) const {
    NJson::TJsonValue reply = NJson::JSON_NULL;
    TString uri = request.GetUri();
    auto replyIt = RepliesMap.find(operationType);
    if (replyIt == RepliesMap.end()) {
        Errors.AddMessage("TestWhsdClient SendRequest", "no suitable reply found");
        return NThreading::MakeErrorFuture<NJson::TJsonValue>(std::make_exception_ptr(yexception() << "No reply found"));
    }
    ui32 replyNum = CurrentReplyNums[operationType]++;
    if (replyNum >= replyIt->second.size()) {
        Errors.AddMessage("TestWhsdClient SendRequest", "too many requests");
        return NThreading::MakeErrorFuture<NJson::TJsonValue>(std::make_exception_ptr(yexception() << "Too many requests"));
    }
    reply = replyIt->second[replyNum];
    VisitedUris.push_back(uri);
    if (reply == NJson::JSON_NULL) {
        return NThreading::MakeErrorFuture<NJson::TJsonValue>(std::make_exception_ptr(yexception() << "Bad request"));
    } else {
        return NThreading::MakeFuture(reply);
    }
}
