#pragma once

#include "config.h"

#include <library/cpp/json/json_value.h>

#include <rtline/library/deprecated/async_impl/client.h>
#include <rtline/util/types/messages_collector.h>

#include <util/generic/maybe.h>

enum EWhsdOperationType {
    GetTransponderEvents /* "get_transponder_events" */,
    OpenSession /* "open_session" */,
    ParseEvent /* "parse_event" */,
    CloseSession /* close_session */,
    CheckWhsdId /* check_whsd_id */
};

class TWhsdTransponderEvent {
    R_FIELD(TString, Id);
    R_FIELD(TString, TransponderId);
    R_FIELD(TString, EntryLocation);
    R_FIELD(TString, ExitLocation);
    R_FIELD(i32, Amount, 0);
    R_FIELD(TInstant, Date);

public:
    bool DeserializeFromJson(const NJson::TJsonValue& jsonEvent, TMessagesCollector& errors);
};

class TWhsdRequestResult {
    R_FIELD(TVector<TWhsdTransponderEvent>, TransponderEvents);
    R_FIELD(NThreading::TFuture<TWhsdRequestResult>, NextPageRequest);

public:
    TWhsdRequestResult(const TVector<TWhsdTransponderEvent>& events)
        : TransponderEvents(events)
    {
    }

    TWhsdRequestResult(const TVector<TWhsdTransponderEvent>& events, NThreading::TFuture<TWhsdRequestResult>&& nextPageRequest)
        : TransponderEvents(events)
        , NextPageRequest(std::move(nextPageRequest))
    {
    }
};

class IWhsdClient {
public:
    virtual NThreading::TFuture<TWhsdRequestResult> GetTransponderEvents(const TInstant from, const TInstant to) const = 0;

    virtual ~IWhsdClient() = default;
};

class TWhsdClient : public IWhsdClient {
public:
    TWhsdClient(const TWhsdClientConfig& config)
        : Config(config)
        , Logger(TRequestLogger<EWhsdOperationType>("whsd-api"))
    {
        Agent = MakeHolder<NNeh::THttpClient>(Config.GetUri(), Config.GetRequestConfig());
    }

    virtual NThreading::TFuture<TWhsdRequestResult> GetTransponderEvents(const TInstant from, const TInstant to) const override;

    const TMessagesCollector& GetErrors() const {
        return Errors;
    }

protected:
    TWhsdClient(const TWhsdClientConfig& config, bool agent)
        : Config(config)
        , Logger(TRequestLogger<EWhsdOperationType>("whsd-api"))
    {
        if (agent) {
            Agent = MakeHolder<NNeh::THttpClient>(config.GetUri(), config.GetRequestConfig());
        }
    }
    virtual NThreading::TFuture<NJson::TJsonValue> SendRequest(EWhsdOperationType operationType, const NNeh::THttpRequest& request) const;
    virtual void LogError(const EWhsdOperationType& type, const TString& message = "") const;

private:
    NThreading::TFuture<TWhsdRequestResult> GetTransponderEvents(const TInstant from, const TInstant to, const ui32 rowsToSkip, bool tryReAuth) const;
    bool Authorize(const bool refreshToken = false) const;
    void CloseSession() const;
    NNeh::THttpRequest CreateRequest(const TString& from, const TString& to, const ui32 rowsSkip, const ui32 rowsLimit) const;

protected:
    mutable TMessagesCollector Errors;
private:
    THolder<NNeh::THttpClient> Agent;
    const TWhsdClientConfig& Config;
    TRequestLogger<EWhsdOperationType> Logger;
    mutable TString Token;
};

class TTestWhsdClient : public TWhsdClient {
    using TOperationRepliesMap = TMap<EWhsdOperationType, TVector<NJson::TJsonValue>>;
    R_FIELD(TOperationRepliesMap, RepliesMap);
    R_FIELD(TVector<TString>, VisitedUris, {}, mutable);

public:
    TTestWhsdClient(const TWhsdClientConfig& config)
        : TWhsdClient(config, false)
    {
    }

protected:
    virtual NThreading::TFuture<NJson::TJsonValue> SendRequest(EWhsdOperationType operationType, const NNeh::THttpRequest& request) const override;
    virtual void LogError(const EWhsdOperationType& type, const TString& message = "") const override;

private:
    mutable TMap<EWhsdOperationType, ui32> CurrentReplyNums;
};
