#include "client.h"

#include <library/cpp/json/json_reader.h>
#include <library/cpp/testing/unittest/registar.h>
#include <util/stream/str.h>


Y_UNIT_TEST_SUITE(WhsdClient) {
    TString GetConfigString() {
        return R"(
            Uri: https://mcabinet.nch-spb.com:443/
            Login: login
            RowsToRead: 50
            <RequestConfig>
                MaxAttempts: 1
            </RequestConfig>
            RequestTimeout: 20s
        )";
    }

    Y_UNIT_TEST(WhsdCorrectReply) {
        const TString replyString = R"({
        "more_rows": 0,
        "return": [
            {
                "remark": "",
                "entry_place": "Приморский пр.",
                "event_type": "1",
                "contract_num": "0514031-KT001",
                "lon": "",
                "class": "1",
                "cdt": "2020.05.09 00:37:20",
                "claim_type": "1",
                "amount": "-17.4",
                "claim_msg": "Оспорить проезд стоимостью 17.4 руб., 2020.05.09 00:37:20, от Приморский пр. до Богатырский пр., 1 класс ТС",
                "place": "Богатырский пр.",
                "entry_date": "2020-05-09T00:37:20",
                "lat": "",
                "rid": "20200509000677246034",
                "receipt_link": "",
                "dt": "2020-05-09T00:37:20",
                "pan": "6362875000008096463",
                "contract_id": "2500910000000000436940"
            },
            {
                "remark": "",
                "entry_place": "наб. Макарова",
                "event_type": "1",
                "contract_num": "0514031-KT001",
                "lon": "",
                "class": "1",
                "cdt": "2020.05.08 19:33:04",
                "claim_type": "1",
                "amount": "-81.79",
                "claim_msg": "Оспорить проезд стоимостью 81.7 руб., 2020.05.08 19:33:04, от наб. Макарова до Краснопутиловская ул., 1 класс ТС",
                "place": "Краснопутиловская ул.",
                "entry_date": "2020-05-08T19:33:04",
                "lat": "",
                "rid": "20200508190677218647",
                "receipt_link": "",
                "dt": "2020-05-08T19:33:04",
                "pan": "6362875000009358466",
                "contract_id": "2500910000000000436940"
            }]
        })";
        NJson::TJsonValue replyJson;
        UNIT_ASSERT(ReadJsonFastTree(replyString, &replyJson));

        const TString authString = R"({ "return": "543778868:CeSgLVzW5KGx2N2UNkRFEc8T2cCyYg8o" })";
        NJson::TJsonValue authJson;
        UNIT_ASSERT(ReadJsonFastTree(authString, &authJson));
        TMap<EWhsdOperationType, TVector<NJson::TJsonValue>> results = {{EWhsdOperationType::OpenSession, {authJson}}, {EWhsdOperationType::GetTransponderEvents, {replyJson}}};

        const TString configString = GetConfigString();
        auto config = TWhsdClientConfig::ParseFromString(configString);
        config.SetPassword("pass");

        TTestWhsdClient whsd(config);
        whsd.SetRepliesMap(results);

        TInstant from = TInstant::ParseIso8601("2020-05-01T07:00:00.000000+00:00");
        TInstant to = TInstant::ParseIso8601("2020-05-02T07:00:00.000000+00:00");
        auto f = whsd.GetTransponderEvents(from, to);
        UNIT_ASSERT_C(f.Initialized() && f.HasValue(), whsd.GetErrors().GetStringReport());
        auto events = f.GetValueSync().GetTransponderEvents();

        UNIT_ASSERT_VALUES_EQUAL(whsd.GetVisitedUris().size(), 2);
        UNIT_ASSERT_VALUES_EQUAL(whsd.GetVisitedUris()[0], "/onyma/system/api/json?function=open_session&user=login&pass=pass&realm=WHSD");
        const TString requestUri = R"(/onyma/system/api/json?function=onm_api_toll_api_mobile_wall&auth_token=543778868:CeSgLVzW5KGx2N2UNkRFEc8T2cCyYg8o&rows_limit=50&rows_skip=0&event_type={"in":[1]}&dt={"bw":["2020-05-01T10:00:00","2020-05-02T10:00:00"]})";
        UNIT_ASSERT_VALUES_EQUAL(whsd.GetVisitedUris()[1], requestUri);

        UNIT_ASSERT_VALUES_EQUAL(events.size(), 2);
        UNIT_ASSERT_VALUES_EQUAL(events[0].GetId(), "20200509000677246034");
        UNIT_ASSERT_VALUES_EQUAL(events[0].GetTransponderId(), "6362875000008096463");
        UNIT_ASSERT_VALUES_EQUAL(events[0].GetEntryLocation(), "Приморский пр.");
        UNIT_ASSERT_VALUES_EQUAL(events[0].GetExitLocation(), "Богатырский пр.");
        UNIT_ASSERT_VALUES_EQUAL(events[0].GetAmount(), -1740);
        UNIT_ASSERT_VALUES_EQUAL(events[0].GetDate(), TInstant::ParseIso8601("2020-05-09T00:37:20.000000+03:00"));
        UNIT_ASSERT_VALUES_EQUAL(events[1].GetId(), "20200508190677218647");
        UNIT_ASSERT_VALUES_EQUAL(events[1].GetTransponderId(), "6362875000009358466");
        UNIT_ASSERT_VALUES_EQUAL(events[1].GetEntryLocation(), "наб. Макарова");
        UNIT_ASSERT_VALUES_EQUAL(events[1].GetExitLocation(), "Краснопутиловская ул.");
        UNIT_ASSERT_VALUES_EQUAL(events[1].GetAmount(), -8179);
        UNIT_ASSERT_VALUES_EQUAL(events[1].GetDate(), TInstant::ParseIso8601("2020-05-08T19:33:04.000000+03:00"));
        UNIT_ASSERT_C(!whsd.GetErrors().HasMessages(), whsd.GetErrors().GetStringReport());
    }

    Y_UNIT_TEST(WhsdCorrectMultipageReply) {
        const TString replyString1 = R"({
        "more_rows": 1,
        "return": [
            {
                "remark": "",
                "entry_place": "Приморский пр.",
                "event_type": "1",
                "contract_num": "0514031-KT001",
                "lon": "",
                "class": "1",
                "cdt": "2020.05.09 00:37:20",
                "claim_type": "1",
                "amount": "-17.4",
                "claim_msg": "Оспорить проезд стоимостью 17.4 руб., 2020.05.09 00:37:20, от Приморский пр. до Богатырский пр., 1 класс ТС",
                "place": "Богатырский пр.",
                "entry_date": "2020-05-09T00:37:20",
                "lat": "",
                "rid": "20200509000677246034",
                "receipt_link": "",
                "dt": "2020-05-09T00:37:20",
                "pan": "6362875000008096463",
                "contract_id": "2500910000000000436940"
            },
            {
                "remark": "",
                "entry_place": "наб. Макарова",
                "event_type": "1",
                "contract_num": "0514031-KT001",
                "lon": "",
                "class": "1",
                "cdt": "2020.05.08 19:33:04",
                "claim_type": "1",
                "amount": "-81.7",
                "claim_msg": "Оспорить проезд стоимостью 81.7 руб., 2020.05.08 19:33:04, от наб. Макарова до Краснопутиловская ул., 1 класс ТС",
                "place": "Краснопутиловская ул.",
                "entry_date": "2020-05-08T19:33:04",
                "lat": "",
                "rid": "20200508190677218647",
                "receipt_link": "",
                "dt": "2020-05-08T19:33:04",
                "pan": "6362875000009358466",
                "contract_id": "2500910000000000436940"
            }]
        })";
        const TString replyString2 = R"({
        "more_rows": 0,
        "return": [
            {
                "remark": "",
                "entry_place": "наб. Макарова",
                "event_type": "1",
                "contract_num": "0514031-KT001",
                "lon": "",
                "class": "1",
                "cdt": "2020.05.08 18:22:25",
                "claim_type": "1",
                "amount": "-99.3",
                "claim_msg": "Оспорить проезд стоимостью 99.3 руб., 2020.05.08 18:22:25, от наб. Макарова до Богатырский пр., 1 класс ТС",
                "place": "Богатырский пр.",
                "entry_date": "2020-05-08T18:22:25",
                "lat": "",
                "rid": "20200508180677166985",
                "receipt_link": "",
                "dt": "2020-05-08T18:22:25",
                "pan": "6362875000007500820",
                "contract_id": "2500910000000000436940"
            }]
        })";
        NJson::TJsonValue replyJson1;
        UNIT_ASSERT(ReadJsonFastTree(replyString1, &replyJson1));
        NJson::TJsonValue replyJson2;
        UNIT_ASSERT(ReadJsonFastTree(replyString2, &replyJson2));

        const TString authString = R"({ "return": "543778868:CeSgLVzW5KGx2N2UNkRFEc8T2cCyYg8o" })";
        NJson::TJsonValue authJson;
        UNIT_ASSERT(ReadJsonFastTree(authString, &authJson));
        TMap<EWhsdOperationType, TVector<NJson::TJsonValue>> results = {{EWhsdOperationType::OpenSession, {authJson}}, {EWhsdOperationType::GetTransponderEvents, {replyJson1, replyJson2}}};

        const TString configString = GetConfigString();
        auto config = TWhsdClientConfig::ParseFromString(configString);
        config.SetPassword("pass");

        TTestWhsdClient whsd(config);
        whsd.SetRepliesMap(results);
        TInstant from = TInstant::ParseIso8601("2020-05-01T07:00:00.000000+00:00");
        TInstant to = TInstant::ParseIso8601("2020-05-02T07:00:00.000000+00:00");
        auto f = whsd.GetTransponderEvents(from, to);
        UNIT_ASSERT_C(f.Initialized() && f.HasValue() && !f.HasException(), whsd.GetErrors().GetStringReport());
        auto result = f.GetValueSync();
        auto events = result.GetTransponderEvents();

        UNIT_ASSERT_VALUES_EQUAL(events.size(), 2);
        UNIT_ASSERT_VALUES_EQUAL(events[0].GetTransponderId(), "6362875000008096463");
        UNIT_ASSERT_VALUES_EQUAL(events[1].GetTransponderId(), "6362875000009358466");

        auto f2 = result.GetNextPageRequest();
        UNIT_ASSERT_C(f2.Initialized() && f2.HasValue() && !f2.HasException(), whsd.GetErrors().GetStringReport());
        UNIT_ASSERT_VALUES_EQUAL(whsd.GetVisitedUris().size(), 3);
        auto result2 = f2.GetValueSync();
        auto events2 = result2.GetTransponderEvents();
        UNIT_ASSERT_VALUES_EQUAL(events2.size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(events2[0].GetTransponderId(), "6362875000007500820");

        UNIT_ASSERT_VALUES_EQUAL(whsd.GetVisitedUris()[0], "/onyma/system/api/json?function=open_session&user=login&pass=pass&realm=WHSD");
        const TString requestUri1 = R"(/onyma/system/api/json?function=onm_api_toll_api_mobile_wall&auth_token=543778868:CeSgLVzW5KGx2N2UNkRFEc8T2cCyYg8o&rows_limit=50&rows_skip=0&event_type={"in":[1]}&dt={"bw":["2020-05-01T10:00:00","2020-05-02T10:00:00"]})";
        UNIT_ASSERT_VALUES_EQUAL(whsd.GetVisitedUris()[1], requestUri1);
        const TString requestUri2 = R"(/onyma/system/api/json?function=onm_api_toll_api_mobile_wall&auth_token=543778868:CeSgLVzW5KGx2N2UNkRFEc8T2cCyYg8o&rows_limit=50&rows_skip=50&event_type={"in":[1]}&dt={"bw":["2020-05-01T10:00:00","2020-05-02T10:00:00"]})";
        UNIT_ASSERT_VALUES_EQUAL(whsd.GetVisitedUris()[2], requestUri2);
        UNIT_ASSERT_C(!whsd.GetErrors().HasMessages(), whsd.GetErrors().GetStringReport());
    }

    Y_UNIT_TEST(WhsdRenewToken) {
        const TString replyString = R"({
        "more_rows": 0,
        "return": [
            {
                "remark": "",
                "entry_place": "Приморский пр.",
                "event_type": "1",
                "contract_num": "0514031-KT001",
                "lon": "",
                "class": "1",
                "cdt": "2020.05.09 00:37:20",
                "claim_type": "1",
                "amount": "-17.4",
                "claim_msg": "Оспорить проезд стоимостью 17.4 руб., 2020.05.09 00:37:20, от Приморский пр. до Богатырский пр., 1 класс ТС",
                "place": "Богатырский пр.",
                "entry_date": "2020-05-09T00:37:20",
                "lat": "",
                "rid": "20200509000677246034",
                "receipt_link": "",
                "dt": "2020-05-09T00:37:20",
                "pan": "6362875000008096463",
                "contract_id": "2500910000000000436940"
            }]
        })";
        NJson::TJsonValue replyJson;
        UNIT_ASSERT(ReadJsonFastTree(replyString, &replyJson));

        const TString authString1 = R"({ "return": "543778868:CeSgLVzW5KGx2N2UNkRFEc8T2cCyYg8o" })";
        NJson::TJsonValue authJson1;
        UNIT_ASSERT(ReadJsonFastTree(authString1, &authJson1));
        const TString authString2 = R"({ "return": "1234:test" })";
        NJson::TJsonValue authJson2;
        UNIT_ASSERT(ReadJsonFastTree(authString2, &authJson2));
        TMap<EWhsdOperationType, TVector<NJson::TJsonValue>> results = {
            {EWhsdOperationType::OpenSession, {authJson1, authJson2}},
            {EWhsdOperationType::GetTransponderEvents, {NJson::JSON_NULL, replyJson}}
        };
        const TString configString = GetConfigString();
        auto config = TWhsdClientConfig::ParseFromString(configString);
        config.SetPassword("pass");

        TTestWhsdClient whsd(config);
        whsd.SetRepliesMap(results);
        TInstant from = TInstant::ParseIso8601("2020-05-01T07:00:00.000000+00:00");
        TInstant to = TInstant::ParseIso8601("2020-05-02T07:00:00.000000+00:00");

        auto f = whsd.GetTransponderEvents(from, to);
        UNIT_ASSERT_C(f.Initialized() && f.HasValue(), whsd.GetErrors().GetStringReport());
        auto events = f.GetValueSync().GetTransponderEvents();

        UNIT_ASSERT_VALUES_EQUAL(whsd.GetVisitedUris().size(), 4);
        UNIT_ASSERT_VALUES_EQUAL(whsd.GetVisitedUris()[0], "/onyma/system/api/json?function=open_session&user=login&pass=pass&realm=WHSD");
        const TString requestUri = R"(/onyma/system/api/json?function=onm_api_toll_api_mobile_wall&auth_token=543778868:CeSgLVzW5KGx2N2UNkRFEc8T2cCyYg8o&rows_limit=50&rows_skip=0&event_type={"in":[1]}&dt={"bw":["2020-05-01T10:00:00","2020-05-02T10:00:00"]})";
        UNIT_ASSERT_VALUES_EQUAL(whsd.GetVisitedUris()[1], requestUri);
        UNIT_ASSERT_VALUES_EQUAL(whsd.GetVisitedUris()[2], "/onyma/system/api/json?function=open_session&user=login&pass=pass&realm=WHSD");
        const TString requestUri2 = R"(/onyma/system/api/json?function=onm_api_toll_api_mobile_wall&auth_token=1234:test&rows_limit=50&rows_skip=0&event_type={"in":[1]}&dt={"bw":["2020-05-01T10:00:00","2020-05-02T10:00:00"]})";
        UNIT_ASSERT_VALUES_EQUAL(whsd.GetVisitedUris()[3], requestUri2);

        UNIT_ASSERT_VALUES_EQUAL(events.size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(events[0].GetTransponderId(), "6362875000008096463");
        UNIT_ASSERT_C(!whsd.GetErrors().HasMessages(), whsd.GetErrors().GetStringReport());
    }

    Y_UNIT_TEST(WhsdRenewTokenBad) {
        const TString authString1 = R"({ "return": "0000:test" })";
        NJson::TJsonValue authJson1;
        UNIT_ASSERT(ReadJsonFastTree(authString1, &authJson1));
        const TString authString2 = R"({ "return": "1234:test" })";
        NJson::TJsonValue authJson2;
        UNIT_ASSERT(ReadJsonFastTree(authString2, &authJson2));
        TMap<EWhsdOperationType, TVector<NJson::TJsonValue>> results = {
            {EWhsdOperationType::OpenSession, {authJson1, authJson2}},
            {EWhsdOperationType::GetTransponderEvents, {NJson::JSON_NULL, NJson::JSON_NULL}}
        };

        const TString configString = GetConfigString();
        auto config = TWhsdClientConfig::ParseFromString(configString);
        config.SetPassword("pass");

        TTestWhsdClient whsd(config);
        whsd.SetRepliesMap(results);

        TInstant from = TInstant::ParseIso8601("2020-05-01T07:00:00.000000+00:00");
        TInstant to = TInstant::ParseIso8601("2020-05-02T07:00:00.000000+00:00");

        auto f = whsd.GetTransponderEvents(from, to);
        UNIT_ASSERT_C(f.HasException() && !f.HasValue(), whsd.GetErrors().GetStringReport());

        UNIT_ASSERT_VALUES_EQUAL(whsd.GetVisitedUris().size(), 4);
        UNIT_ASSERT_VALUES_EQUAL(whsd.GetVisitedUris()[0], "/onyma/system/api/json?function=open_session&user=login&pass=pass&realm=WHSD");
        const TString requestUri = R"(/onyma/system/api/json?function=onm_api_toll_api_mobile_wall&auth_token=0000:test&rows_limit=50&rows_skip=0&event_type={"in":[1]}&dt={"bw":["2020-05-01T10:00:00","2020-05-02T10:00:00"]})";
        UNIT_ASSERT_VALUES_EQUAL(whsd.GetVisitedUris()[1], requestUri);
        UNIT_ASSERT_VALUES_EQUAL(whsd.GetVisitedUris()[2], "/onyma/system/api/json?function=open_session&user=login&pass=pass&realm=WHSD");
        const TString requestUri2 = R"(/onyma/system/api/json?function=onm_api_toll_api_mobile_wall&auth_token=1234:test&rows_limit=50&rows_skip=0&event_type={"in":[1]}&dt={"bw":["2020-05-01T10:00:00","2020-05-02T10:00:00"]})";
        UNIT_ASSERT_VALUES_EQUAL(whsd.GetVisitedUris()[3], requestUri2);

        UNIT_ASSERT_C(!whsd.GetErrors().HasMessages(), whsd.GetErrors().GetStringReport());
    }

    Y_UNIT_TEST(WhsdNoLogin) {
        const TString authString1 = R"({ "return": "0000:test" })";
        NJson::TJsonValue authJson1;
        UNIT_ASSERT(ReadJsonFastTree(authString1, &authJson1));
        const TString authString2 = R"({ "return": "1234:test" })";
        NJson::TJsonValue authJson2;
        UNIT_ASSERT(ReadJsonFastTree(authString2, &authJson2));
        TMap<EWhsdOperationType, TVector<NJson::TJsonValue>> results = {
            {EWhsdOperationType::OpenSession, {authJson1, authJson2}},
            {EWhsdOperationType::GetTransponderEvents, {NJson::JSON_NULL, NJson::JSON_NULL}}
        };
        const TString configString = GetConfigString();
        auto config = TWhsdClientConfig::ParseFromString(configString);
        config.SetPassword("pass");
        config.SetLogin("");

        TTestWhsdClient whsd(config);
        whsd.SetRepliesMap(results);

        TInstant from = TInstant::ParseIso8601("2020-05-01T07:00:00.000000+00:00");
        TInstant to = TInstant::ParseIso8601("2020-05-02T07:00:00.000000+00:00");

        auto f = whsd.GetTransponderEvents(from, to);
        UNIT_ASSERT(f.HasException());
        UNIT_ASSERT(whsd.GetErrors().HasMessages());

        UNIT_ASSERT_VALUES_EQUAL(whsd.GetVisitedUris().size(), 0);
    }
}
