#include "client.h"

#include <drive/library/cpp/threading/future.h>
#include <drive/library/cpp/raw_text/datetime.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/json/field.h>
#include <rtline/library/json/merge.h>
#include <rtline/util/algorithm/container.h>

#include <util/string/builder.h>
#include <util/string/join.h>
#include <library/cpp/json/json_reader.h>

template <>
NJson::TJsonValue NJson::ToJson(const TYaDocClient::TDocument& object) {
    NJson::TJsonValue result;
    NJson::InsertField(result, "doc_id", object.Id);
    NJson::InsertField(result, "doc_type", object.Type);
    NJson::InsertField(result, "doc_number", object.Number);
    NJson::InsertField(result, "doc_date", object.Date);
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TYaDocClient::TDocument& result) {
    return NJson::ParseField(value, "doc_id", result.Id, true)
        && NJson::ParseField(value, "doc_type", result.Type)
        && NJson::ParseField(value, "doc_number", result.Number)
        && NJson::ParseField(value, "doc_date", result.Date);
}

NThreading::TFuture<TYaDocClient::TDocuments> TYaDocClient::GetDocuments(TInstant fromTs, TInstant toTs, const TString& contractId) const {
    if (!TvmClient) {
        throw yexception() << "TvmClient not configured";
    }
    TVector<NThreading::TFuture<TYaDocClient::TDocuments>> responses;
    TInstant toRequestTs = fromTs;
    TInstant fromRequestTs = fromTs;
    while (toRequestTs < toTs) {
        toRequestTs = Min<TInstant>(toTs, fromRequestTs + Config.GetMaxPeriod());

        NNeh::THttpRequest request;
        request.SetUri(Config.GetPathPrefix() + "documents")
            .AddHeader("Content-Type", "application/json")
            .AddHeader("X-Ya-Service-Ticket", TvmClient->GetServiceTicketFor(Config.GetDestinationTvmId()))
            .SetPostData(NJson::TMapBuilder("date_from", NUtil::FormatDatetime(fromRequestTs, "%Y-%m-%d"))("date_to", NUtil::FormatDatetime(toRequestTs, "%Y-%m-%d"))("contract_id", contractId)("exclude_reversed", true));

        auto response = Client->SendAsync(request, Now() + Config.GetRequestTimeout()).Apply([contractId](const NThreading::TFuture<NUtil::THttpReply>& r) {
            const auto& reply = r.GetValue();
            if (!reply.IsSuccessReply()) {
                throw yexception() << "Request error for " << contractId << ", reply code " << reply.Code() << ", error: " << reply.ErrorMessage() << ", content: " << reply.Content();
            }
            NJson::TJsonValue replyJson;
            if (!NJson::ReadJsonFastTree(reply.Content(), &replyJson)) {
                throw yexception() << "Cannot parse reply json " << reply.Content();
            }

            TDocuments documents;
            if (!NJson::TryFromJson(replyJson["documents"], documents)) {
                throw yexception() << "Cannot extract data " << reply.Content();
            }
            return NThreading::MakeFuture(documents);
        });
        responses.emplace_back(std::move(response));
        fromRequestTs = TInstant::Days(toRequestTs.Days()) + TDuration::Days(1);
    }

    return NThreading::WaitExceptionOrAll(responses).Apply([responses = std::move(responses)](const NThreading::TFuture<void>& r) -> NThreading::TFuture<TYaDocClient::TDocuments> {
        if (!r.HasValue()) {
            return NThreading::MakeErrorFuture<TYaDocClient::TDocuments>(NThreading::GetException(r));
        }

        TMap<ui64, TYaDocClient::TDocument> documents;
        for (auto&& response : responses) {
            for (auto&& doc : response.GetValue()) {
                documents.emplace(doc.Id, std::move(doc));
            }
        }
        return NThreading::MakeFuture(MakeVector(NContainer::Values(documents)));
    });
}

TYaDocClient::TRawDocument::TRawDocument(const NUtil::THttpReply& reply)
    : Buffer(reply.Content())
{
    MimeType = "application/octet-stream";
    auto contentHeader = reply.GetHeaders().FindHeader("Content-Type");
    if (contentHeader) {
        MimeType = contentHeader->Value();
    }
}

NThreading::TFuture<TYaDocClient::TRawDocument> TYaDocClient::GetBinaryFile(const TString& uri) const {
    if (!TvmClient) {
        throw yexception() << "TvmClient not configured";
    }

    NNeh::THttpRequest request;
    request.SetUri(uri)
        .SetRequestType("GET")
        .AddHeader("X-Ya-Service-Ticket", TvmClient->GetServiceTicketFor(Config.GetDestinationTvmId()));

    auto asyncResult = Client->SendAsync(request, Now() + Config.GetRequestTimeout());
    return asyncResult.Apply([uri](const NThreading::TFuture<NUtil::THttpReply>& r) -> NThreading::TFuture<TYaDocClient::TRawDocument> {
        if (r.HasException() || !r.HasValue()) {
            throw NThreading::GetException(r);
        }

        const auto& reply = r.GetValue();
        if (!reply.IsSuccessReply()) {
            throw yexception() << "Request error for " << uri << ", reply code " << reply.Code() << ", error: " << reply.ErrorMessage() << ", content: " << reply.Content();
        };
        return NThreading::MakeFuture(TRawDocument(reply));
    });
}

NThreading::TFuture<TYaDocClient::TRawDocument> TYaDocClient::GetDocumentById(const TString& id) const {
    return GetBinaryFile(Config.GetPathPrefix() + "documents/" + id + "/download");
}

NThreading::TFuture<TYaDocClient::TRawDocument> TYaDocClient::GetArchive(const TVector<TString>& ids) const {
    return GetBinaryFile(Config.GetPathPrefix() + "documents/download?documentIds=" + JoinSeq(",", ids));
}
