#include "client.h"

#include <drive/library/cpp/threading/future.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/json/field.h>
#include <rtline/library/json/merge.h>

#include <library/cpp/json/json_reader.h>

NThreading::TFuture<NJson::TJsonValue> TYangClient::RequestAssignmentAsync(const TString& assignmentId) const {
    auto uri = Config.GetPathPrefix() + "assignments/" + assignmentId;

    NNeh::THttpRequest request;
    request.SetUri(uri)
        .SetRequestType("GET")
        .AddHeader("Content-Type", "application/json");

    auto replyAsync = RequestAsync(request);
    return replyAsync.Apply([uri = request.GetUri()](const NThreading::TFuture<NUtil::THttpReply>& r) -> NThreading::TFuture<NJson::TJsonValue> {
        NJson::TJsonValue replyJson;
        const auto& reply = r.GetValue();
        if (!NJson::ReadJsonFastTree(reply.Content(), &replyJson)) {
            throw TYangJsonParseException() << reply.Content();
        }

        return NThreading::MakeFuture(replyJson);
    });
}

NJson::TJsonValue TYangClient::RequestAssignmentSync(const TString& assignmentId) const {
    auto assignmentAsync = RequestAssignmentAsync(assignmentId);
    return assignmentAsync.GetValueSync();
}

NThreading::TFuture<NNeh::THttpReply> TYangClient::RequestAsync(NNeh::THttpRequest& request) const {
    ApplyAuth(request);

    auto replyAsync = Client->SendAsync(request, Now() + Config.GetRequestTimeout());
    return replyAsync.Apply([uri = request.GetUri()](const NThreading::TFuture<NUtil::THttpReply>& r) -> NThreading::TFuture<NNeh::THttpReply> {
        const auto& reply = r.GetValue();

        if (reply.Code() == 404) {
            throw TYangNotFoundException() << "Request error for " << uri << ", reply code " << reply.Code() << ", error: " << reply.ErrorMessage() << ", content: " << reply.Content();
        };

        if (!reply.IsSuccessReply()) {
            throw TYangRequestError() << "Request error for " << uri << ", reply code " << reply.Code() << ", error: " << reply.ErrorMessage() << ", content: " << reply.Content();
        };
        return NThreading::MakeFuture(reply);
    });
}

NNeh::THttpReply TYangClient::RequestSync(NNeh::THttpRequest& request) const {
    auto replyAsync = RequestAsync(request);
    return replyAsync.GetValueSync();
}

void TYangClient::ApplyAuth(NNeh::THttpRequest& request) const {
    switch (Config.GetAuthType()) {
        case TYangClientConfig::EAuthType::OAuth:
            request.AddHeader("Authorization", "OAuth " + Config.GetAuthToken());
            break;
        case TYangClientConfig::EAuthType::TVM:
            if (!TvmClient) {
                throw TYangAuthException() << "TvmClient's not configured";
            }
            request.AddHeader("X-Ya-Service-Ticket", TvmClient->GetServiceTicketFor(Config.GetDestinationTvmId()));
            break;
        default:
            throw TYangAuthException() << "Unknown AuthType. Use OAuth or TVM instead";
    }
}

namespace {
    TMutex FakeClientMutex;
}

NThreading::TFuture<NJson::TJsonValue> TFakeYangClient::RequestAssignmentAsync(const TString& assignmentId) const {
    TGuard<TMutex> g(FakeClientMutex);
    return NThreading::MakeFuture(assignments[assignmentId]);
}

NJson::TJsonValue TFakeYangClient::RequestAssignmentSync(const TString& assignmentId) const {
    TGuard<TMutex> g(FakeClientMutex);
    return assignments[assignmentId];
}

void TFakeYangClient::InsertAssignment(const TString& assignmentId, const NJson::TJsonValue& assignment) const {
    TGuard<TMutex> g(FakeClientMutex);
    assignments[assignmentId] = assignment;
}
