#include "config.h"

#include <util/stream/file.h>

void TYangClientConfig::Init(const TYandexConfig::Section* section) {
    auto directives = section->GetDirectives();
    Uri = directives.Value<TString>("Uri", Uri);
    PathPrefix = directives.Value<TString>("PathPrefix", PathPrefix);

    AuthType = directives.Value<EAuthType>("AuthType", AuthType);
    switch (AuthType) {
        case EAuthType::OAuth:
            AuthToken = directives.Value<TString>("AuthToken", AuthToken);
            TokenPath = directives.Value<TString>("TokenPath", TokenPath);
            if (!AuthToken && TokenPath) {
                AuthToken = Strip(TIFStream(TokenPath).ReadAll());
            }
            if (!AuthToken) {
                throw yexception() << "AuthToken is undefined";
            }
            break;
        case EAuthType::TVM:
            SelfTvmId = directives.Value<ui32>("SelfTvmId", SelfTvmId);
            DestinationTvmId = directives.Value<ui32>("DestinationTvmId", DestinationTvmId);
            if (!SelfTvmId) {
                throw yexception() << "SelfTvmId is undefined";
            }
            if (!DestinationTvmId) {
                throw yexception() << "DestinationTvmId is undefined";
            }
            break;
        default:
            throw yexception() << "Undefined AuthType. OAuth or TVM are required";
    }

    RequestTimeout = directives.Value<TDuration>("RequestTimeout", RequestTimeout);
    const TYandexConfig::TSectionsMap children = section->GetAllChildren();
    auto it = children.find("RequestConfig");
    if (it != children.end()) {
        RequestConfig.InitFromSection(it->second);
    }
}

void TYangClientConfig::ToString(IOutputStream& os) const {
    os << "Uri: " << Uri << Endl;

    os << "AuthType: " << ::ToString(AuthType) << Endl;
    os << "TokenPath: " << TokenPath << Endl;
    os << "PathPrefix: " << PathPrefix << Endl;
    os << "SelfTvmId: " << SelfTvmId << Endl;
    os << "DestinationTvmId: " << DestinationTvmId << Endl;
    os << "RequestTimeout: " << RequestTimeout << Endl;
    os << "<RequestConfig>" << Endl;
    RequestConfig.ToString(os);
    os << "</RequestConfig>" << Endl;
}

TYangClientConfig TYangClientConfig::ParseFromString(const TString& configStr) {
    TYangClientConfig result;
    TAnyYandexConfig config;
    CHECK_WITH_LOG(config.ParseMemory(configStr.data()));
    result.Init(config.GetRootSection());
    return result;
}
