#include <drive/library/cpp/yang/client.h>

#include <library/cpp/testing/unittest/tests_data.h>
#include <library/cpp/testing/unittest/registar.h>

#include <util/system/env.h>

TYangClientConfig GetClientConfig() {
    TString uri = GetEnv("YangUri");
    UNIT_ASSERT_C(uri, "env YangUri undefined");
    TString authToken = GetEnv("AuthToken");
    UNIT_ASSERT_C(authToken, "env authToken undefined");

    auto config = TStringBuilder()
        << "Uri: " << uri << Endl
        << "AuthType: OAuth" << Endl
        << "AuthToken: " << authToken << Endl
        << "PathPrefix: /api/v1/" << Endl;

    return TYangClientConfig::ParseFromString(config);
}

Y_UNIT_TEST_SUITE(YangSuite) {
    Y_UNIT_TEST(RequestAsync) {
        TYangClientConfig config = GetClientConfig();
        TYangClient client(config);

        NNeh::THttpRequest request;
        request.SetUri("assignments/");

        auto future = client.RequestAsync(request);
        future.Wait();

        UNIT_ASSERT(!future.HasException());
        auto reply = future.GetValue();
        UNIT_ASSERT_VALUES_EQUAL(reply.Code(), 200);
        
        auto server = reply.GetHeaders().FindHeader("Server");
        UNIT_ASSERT(server);
        UNIT_ASSERT_VALUES_EQUAL(server->Value(), "nginx");
    }

    Y_UNIT_TEST(RequestSync) {
        TYangClientConfig config = GetClientConfig();
        TYangClient client(config);

        NNeh::THttpRequest request;
        request.SetUri(config.GetPathPrefix() + "assignments/");
        UNIT_CHECK_GENERATED_EXCEPTION(client.RequestSync(request), TYangRequestError);

        request.SetUri("assignments/");
        auto reply = client.RequestSync(request); 
        UNIT_ASSERT_VALUES_EQUAL(reply.Code(), 200);
        
        auto server = reply.GetHeaders().FindHeader("Server");
        UNIT_ASSERT(server);
        UNIT_ASSERT_VALUES_EQUAL(server->Value(), "nginx");
    }

    Y_UNIT_TEST(AssignmentAsync) {
        TYangClientConfig config = GetClientConfig();
        TYangClient client(config);

        TString wrongAssignmentId = "-1";
        TString assignmentId = GetEnv("AssignmentId");
        UNIT_ASSERT_C(assignmentId, "env AssignmentId undefined");

        auto futureWrongAssignment = client.RequestAssignmentAsync(wrongAssignmentId);
        futureWrongAssignment.Wait();

        UNIT_ASSERT(futureWrongAssignment.HasException());
        UNIT_CHECK_GENERATED_EXCEPTION(futureWrongAssignment.GetValue(), TYangNotFoundException);

        auto futureGoodAssignment = client.RequestAssignmentAsync(assignmentId);
        futureGoodAssignment.Wait();

        UNIT_ASSERT(futureGoodAssignment.HasValue());
        auto assignment = futureGoodAssignment.GetValueSync();
        UNIT_ASSERT(!assignment["id"].GetString().Empty());
    }

    Y_UNIT_TEST(AssignmentSync) {
        TYangClientConfig config = GetClientConfig();
        TYangClient client(config);

        TString wrongAssignmentId = "-1";
        TString assignmentId = GetEnv("AssignmentId");
        UNIT_ASSERT_C(assignmentId, "env AssignmentId undefined");

        UNIT_CHECK_GENERATED_EXCEPTION(client.RequestAssignmentSync(wrongAssignmentId), TYangNotFoundException);

        auto assignment = client.RequestAssignmentSync(assignmentId);
        UNIT_ASSERT(!assignment["id"].GetString().Empty());
    }
}
