package lkmtzs

import (
	"bufio"
	"bytes"
	"fmt"
	"io"
	"io/ioutil"
	"log"
	"net/http"
	"net/url"
	"strconv"
	"strings"
	"time"

	"github.com/PuerkitoBio/goquery"
	"golang.org/x/text/encoding/charmap"
)

const (
	Production = "http://lkmtzs.gtultima.ru"
)

// Client represents Tatneft client
type Client struct {
	endpoint string
	client   http.Client
}

// Session represents client session.
type Session struct {
	JSessionID string
}

func (c *Client) Login(login, password string) (*Session, error) {
	form := url.Values{
		"login": {login},
		"pass":  {password},
		"org":   {"on"},
		"type":  {"inn"},
	}
	req, err := http.NewRequest(
		http.MethodPost, c.endpoint+"/logOn",
		strings.NewReader(form.Encode()),
	)
	if err != nil {
		return nil, err
	}
	req.Header.Set("Content-Type", "application/x-www-form-urlencoded")
	resp, err := c.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer func() {
		if err := resp.Body.Close(); err != nil {
			log.Println("Error:", err)
		}
	}()
	if resp.StatusCode != 302 {
		body, err := ioutil.ReadAll(resp.Body)
		if err != nil {
			return nil, fmt.Errorf("invalid status code: %d", resp.StatusCode)
		}
		return nil, fmt.Errorf(
			"invalid status code %d with content: %s", resp.StatusCode, body,
		)
	}
	session := Session{}
	for _, cookie := range resp.Cookies() {
		if cookie.Name == "JSESSIONID" && cookie.Value != "" {
			session.JSessionID = cookie.Value
		}
	}
	return &session, nil
}

func (c *Client) Logout(s *Session) error {
	req, err := c.newRequest(http.MethodGet, "/logOut", nil, s)
	if err != nil {
		return err
	}
	resp, err := c.client.Do(req)
	if err != nil {
		return err
	}
	defer func() {
		if err := resp.Body.Close(); err != nil {
			log.Println("Error:", err)
		}
	}()
	if resp.StatusCode != 302 {
		return fmt.Errorf("invalid status code: %d", resp.StatusCode)
	}
	return nil
}

type Tx struct {
	ID       string
	Time     time.Time
	Card     string
	Amount   float64
	FuelCode string
}

func (c *Client) RequestTxs(
	s *Session, start, finish time.Time,
) error {
	form := url.Values{}
	form.Set("period", "0")
	form.Set("start", start.Format("2006-01-02T15:04"))
	form.Set("end", finish.Format("2006-01-02T15:04"))
	req, err := c.newRequest(
		http.MethodPost, "/viewStat",
		strings.NewReader(form.Encode()), s,
	)
	if err != nil {
		return err
	}
	if _, err := c.doRequest(req); err != nil {
		return err
	}
	return nil
}

func (c *Client) GetRawTxs(s *Session, typ string) ([]byte, error) {
	query := url.Values{}
	query.Set("type", typ)
	query.Set("role", "partner")
	req, err := c.newRequest(
		http.MethodGet, "/export?"+query.Encode(), nil, s,
	)
	if err != nil {
		return nil, err
	}
	return c.doRequest(req)
}

func (c *Client) RequestTxs2(
	s *Session, start, finish time.Time,
) ([]Tx, error) {
	form := url.Values{}
	form.Set("period", "0")
	form.Set("start", start.Format("2006-01-02T15:04"))
	form.Set("end", finish.Format("2006-01-02T15:04"))
	req, err := c.newRequest(
		http.MethodPost, "/viewStat",
		strings.NewReader(form.Encode()), s,
	)
	if err != nil {
		return nil, err
	}
	data, err := c.doRequest2(req)
	if err != nil {
		return nil, err
	}
	doc, err := goquery.NewDocumentFromReader(bytes.NewBuffer(data))
	if err != nil {
		return nil, err
	}
	var txs []Tx
	doc.Find(".table").Each(func(i int, s *goquery.Selection) {
		var cols []string
		s.Find("th").Each(func(i int, s *goquery.Selection) {
			cols = append(cols, s.Text())
		})
		s.Find("tr").Each(func(i int, s *goquery.Selection) {
			var tx Tx
			var good bool
			s.Find("td").Each(func(i int, s *goquery.Selection) {
				txt := strings.TrimSpace(s.Text())
				switch strings.TrimSpace(cols[i]) {
				case "Дата":
					tx.Time, err = parseRusTime(txt)
				case "Тип операции":
					if txt == "Дебетование с терминала" {
						good = true
					}
				case "Кошелек":
					tx.FuelCode = txt
				case "Кол-во, л.":
					tx.Amount, err = strconv.ParseFloat(txt, 64)
				case "Сумма, руб.":
				case "Карта":
					tx.Card = txt
				case "Номер ТС":
				case "АЗС":
				case "№ заказа":
					tx.ID = txt
				}
			})
			if good {
				txs = append(txs, tx)
			}
		})
	})
	return txs, err
}

func parseRusTime(s string) (time.Time, error) {
	orig := s
	for i, val := range []string{
		"янв", "фев", "мар", "апр", "май", "июн",
		"июл", "авг", "сен", "окт", "ноя", "дек",
		"янв", "фев", "мар", "апр", "мая", "июн",
		"июл", "авг", "сен", "окт", "ноя", "дек",
	} {
		s = strings.Replace(s, val, fmt.Sprintf("%02d", i%12+1), 1)
	}
	ts, err := time.Parse("02 01 2006 15:04", s)
	if err != nil {
		return time.Time{}, fmt.Errorf("parseRusTime: %w", err)
	}
	if s != orig {
		ts = ts.Add(-3 * time.Hour)
	}
	return ts, nil
}

func ParseRawTxs(data []byte) ([]Tx, error) {
	scanner := bufio.NewScanner(bytes.NewReader(data))
	var txs []Tx
	for scanner.Scan() {
		words := strings.Split(scanner.Text(), ";")
		if len(words) < 10 {
			continue
		}
		ts, err := parseRusTime(words[0])
		if err != nil {
			return nil, err
		}
		amount, err := strconv.ParseFloat(words[3], 64)
		if err != nil {
			return nil, err
		}
		txs = append(txs, Tx{
			Time:     ts,
			Card:     words[5],
			Amount:   amount,
			FuelCode: words[2],
		})
	}
	return txs, nil
}

func (c *Client) newRequest(
	method string, url string, body io.Reader, s *Session,
) (*http.Request, error) {
	req, err := http.NewRequest(method, c.endpoint+url, body)
	if err != nil {
		return nil, err
	}
	req.Header.Add("Content-Type", "application/x-www-form-urlencoded")
	if s != nil {
		req.AddCookie(&http.Cookie{Name: "JSESSIONID", Value: s.JSessionID})
	}
	return req, nil
}

func (c *Client) doRequest(req *http.Request) ([]byte, error) {
	resp, err := c.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer func() {
		if err := resp.Body.Close(); err != nil {
			log.Println("Error:", err)
		}
	}()
	decoder := charmap.Windows1251.NewDecoder()
	body, err := ioutil.ReadAll(decoder.Reader(resp.Body))
	if err != nil {
		return nil, err
	}
	if resp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf(
			"wrong response status %q with content %q",
			resp.Status, body,
		)
	}
	return body, nil
}

func (c *Client) doRequest2(req *http.Request) ([]byte, error) {
	resp, err := c.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer func() {
		if err := resp.Body.Close(); err != nil {
			log.Println("Error:", err)
		}
	}()
	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, err
	}
	if resp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf(
			"wrong response status %q with content %q",
			resp.Status, body,
		)
	}
	return body, nil
}

// NewClient creates a new instance of Tatneft client.
//
// You can pass to endpoint "", if you want to use production endpoint, or
// pass "Production" if you want to specify production endpoint explicitly.
func NewClient(endpoint string) *Client {
	if endpoint == "" || endpoint == "Production" {
		endpoint = Production
	}
	return &Client{
		endpoint: endpoint,
		client: http.Client{
			Timeout: time.Minute,
			CheckRedirect: func(*http.Request, []*http.Request) error {
				return http.ErrUseLastResponse
			},
		},
	}
}
