package renis

import (
	"bytes"
	"encoding/xml"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"time"
)

const (
	Production = "https://services.renins.com/PartnerIntegrationService/IntegrationService.svc"
	Testing    = "http://stg.services.renins.com/PartnerIntegrationService/IntegrationService.svc"
)

// Client represents Tatneft client
type Client struct {
	endpoint   string
	clientName string
	partnerUID string
	client     http.Client
}

// NewClient creates a new instance of Renis client.
//
// You can pass to endpoint "", if you want to use production endpoint, or
// pass "Production" if you want to specify production endpoint explicitly.
func NewClient(endpoint, clientName, partnerUID string) *Client {
	switch endpoint {
	case "", "production":
		endpoint = Production
	case "testing":
		endpoint = Testing
	}
	return &Client{
		endpoint:   endpoint,
		clientName: clientName,
		partnerUID: partnerUID,
		client: http.Client{
			Timeout: 5 * time.Minute,
			CheckRedirect: func(*http.Request, []*http.Request) error {
				return http.ErrUseLastResponse
			},
		},
	}
}

type PhysicalPerson struct {
	FirstName  string
	LastName   string
	MiddleName string
	Serial     string
	Number     string
	ExpDate    time.Time
	BirthDate  time.Time
	Country    string
	Province   string
	Region     string
	City       string
	// Passport info.
	DocFirstName  string
	DocLastName   string
	DocMiddleName string
	DocIssueDate  time.Time
	DocSerial     string
	DocNumber     string
}

type PhysicalPersonKBM struct {
	Serial   string
	Number   string
	KBM      float64
	FirstKBM float64
}

type envelope struct {
	XMLName xml.Name `xml:"s:Envelope"`
	// S contains SOAP url.
	S string `xml:"xmlns:s,attr"`
	// Header contains empty header.
	Header struct{} `xml:"s:Header"`
	// Body contains body.
	Body envelopeBody `xml:"s:Body"`
}

type envelopeResp struct {
	XMLName xml.Name `xml:"Envelope"`
	// Header contains empty header.
	Header struct{} `xml:"Header"`
	// Body contains body.
	Body envelopeBody `xml:"Body"`
}

type envelopeBody struct {
	MakeCalculation         *makeCalculation         `xml:"MakeCalculation"`
	MakeCalculationResponse *makeCalculationResponse `xml:"MakeCalculationResponse"`
}

type makeCalculation struct {
	XMLNS string `xml:"xmlns,attr"`
	// Request contains calc request.
	Request *calcRequest `xml:"doc>Request"`
}

type makeCalculationResponse struct {
	XMLNS string `xml:"xmlns,attr"`
	// Request contains calc request.
	Response *calcResponse `xml:"MakeCalculationResult>root>CalcResponse"`
}

type calcResponse struct {
	Value calcResponseValue `xml:"CalcResponseValue"`
}

type calcResponseValue struct {
	CalcKBMResponses []calcKBMResponse `xml:"CalcKBMResponses>CalcKBMResponse"`
}

type calcKBMResponse struct {
	PersonDocument personDocument `xml:"PersonDocument"`
	KBMValue       float64        `xml:"KBMValue"`
	KBMFirstValue  float64        `xml:"KBMFirstValue"`
}

type calcRequest struct {
	XMLNS            string            `xml:"xmlns,attr"`
	Type             int               `xml:"type,attr"`
	ClientSystemName string            `xml:"ClientSystemName,attr"`
	PartnerUID       string            `xml:"partnerUid,attr"`
	UID              string            `xml:"uid,attr"`
	Policy           calcPolicyRequest `xml:"Policy"`
}

type calcPolicyRequest struct {
	ContractTerm contractTerm  `xml:"ContractTerm"`
	OSAGO        *osago        `xml:"Osago"`
	Vehicle      *vehicle      `xml:"Vehicle"`
	Participants *participants `xml:"Participants"`
}

type vehicle struct {
	Manufacturer     string   `xml:"Manufacturer"`
	Model            string   `xml:"Model"`
	Cost             string   `xml:"Cost"`
	Year             string   `xml:"Year"`
	Power            string   `xml:"Power"`
	ManufacturerType string   `xml:"ManufacturerType"`
	Type             string   `xml:"Type"`
	CarBodyType      string   `xml:"CarBodyType"`
	CarIdent         carIdent `xml:"CarIdent"`
}

type carIdent struct {
	VIN string `xml:"VIN"`
}

type participants struct {
	Drivers  drivers  `xml:"Drivers"`
	Insurant insurant `xml:"Insurant"`
	Owner    owner    `xml:"Owner"`
}

type drivers struct {
	Multidrive string `xml:"Multidrive,attr"`
	Type       int    `xml:"type,attr"`
}

type insurant struct {
	Type int `xml:"type,attr"`
}

type owner struct {
	Type int `xml:"type,attr"`
}

type contractTerm struct {
	Product       int    `xml:"Product"`
	ProgramType   string `xml:"ProgramType"`
	DurationMonth int    `xml:"DurationMonth"`
	Purpose       string `xml:"Purpose"`
	PaymentType   int    `xml:"PaymentType"`
	Currency      string `xml:"Currency"`
}

type osago struct {
	CalculationType     int            `xml:"CalculationType,attr"`
	RegistrationCountry string         `xml:"RegistrationCountry"`
	RegistrationPlace   string         `xml:"RegistrationPlace"`
	KBM                 float32        `xml:"Kbm"`
	CalcKBMRequest      calcKBMRequest `xml:"CalcKBMRequest"`
}

type calcKBMRequest struct {
	DateKBM         string           `xml:"DateKBM"`
	PhysicalPersons []physicalPerson `xml:"PhysicalPersons>PhysicalPerson"`
}

type physicalPerson struct {
	PersonDocument       personDocument `xml:"PersonDocument"`
	DriverDocument       driverDocument `xml:"DriverDocument"`
	PersonSecondName     string         `xml:"PersonSecondName"`
	PersonFirstName      string         `xml:"PersonFirstName"`
	PersonSurName        string         `xml:"PersonSurName"`
	PersonBirthDate      string         `xml:"PersonBirthDate"`
	DriverExperienceDate string         `xml:"DriverExperienceDate"`
	PersonDocSecondName  string         `xml:"PersonDocSecondName"`
	PersonDocFirstName   string         `xml:"PersonDocFirstName"`
	PersonDocSurName     string         `xml:"PersonDocSurName"`
	PersonAddress        personAddress  `xml:"PersonAddress"`
}

type personDocument struct {
	DocPerson   string `xml:"DocPerson"`
	Serial      string `xml:"Serial"`
	Number      string `xml:"Number"`
	IssuedWhere string `xml:"IssuedWhere"`
	IssuedDate  string `xml:"IssuedDate"`
}

type driverDocument struct {
	Serial string `xml:"Serial"`
	Number string `xml:"Number"`
}

type personAddress struct {
	Country  string `xml:"Country"`
	Province string `xml:"Province"`
	Region   string `xml:"Region"`
	City     string `xml:"City"`
}

func (c *Client) CalcKBM(
	persons []PhysicalPerson, reqID string,
) ([]PhysicalPersonKBM, error) {
	var dataPersons []physicalPerson
	for _, person := range persons {
		dataPerson := physicalPerson{
			PersonDocument: personDocument{
				DocPerson:   "PASSPORT",
				Serial:      person.DocSerial,
				Number:      person.DocNumber,
				IssuedWhere: "Тест",
				IssuedDate:  person.DocIssueDate.Format("2006-01-02"),
			},
			DriverDocument: driverDocument{
				Serial: person.Serial,
				Number: person.Number,
			},
			PersonSecondName:     person.MiddleName,
			PersonFirstName:      person.FirstName,
			PersonSurName:        person.LastName,
			PersonBirthDate:      person.BirthDate.Format("2006-01-02"),
			DriverExperienceDate: person.ExpDate.Format("2006-01-02"),
			PersonDocSecondName:  person.DocMiddleName,
			PersonDocFirstName:   person.DocFirstName,
			PersonDocSurName:     person.DocLastName,
			PersonAddress: personAddress{
				Country:  person.Country,
				Province: person.Province,
				Region:   person.Region,
				City:     person.City,
			},
		}
		dataPersons = append(dataPersons, dataPerson)
	}
	data := envelope{
		S: "http://schemas.xmlsoap.org/soap/envelope/",
		Body: envelopeBody{
			MakeCalculation: &makeCalculation{
				XMLNS: "http://renins.com/",
				Request: &calcRequest{
					ClientSystemName: c.clientName,
					PartnerUID:       c.partnerUID,
					UID:              reqID,
					Policy: calcPolicyRequest{
						ContractTerm: contractTerm{
							Product:       3,
							ProgramType:   "OSAGO",
							DurationMonth: 12,
							Purpose:       "аренда/прокат",
							PaymentType:   1,
							Currency:      "RUR",
						},
						OSAGO: &osago{
							CalculationType:     1,
							RegistrationCountry: "Россия",
							RegistrationPlace:   "Москва",
							KBM:                 1,
							CalcKBMRequest: calcKBMRequest{
								DateKBM:         time.Now().Format("2006-01-02T15:04:05"),
								PhysicalPersons: dataPersons,
							},
						},
						Vehicle: &vehicle{
							Manufacturer:     "Nissan",
							Model:            "Qashqai",
							Cost:             "1000000",
							Year:             "2017",
							Power:            "150",
							ManufacturerType: "0",
							Type:             "Легковое ТС",
							CarBodyType:      "Седан",
							CarIdent: carIdent{
								VIN: "12345678901234567",
							},
						},
						Participants: &participants{
							Drivers:  drivers{Multidrive: "NO", Type: 1},
							Insurant: insurant{Type: 1},
							Owner:    owner{Type: 1},
						},
					},
				},
			},
		},
	}
	body, err := xml.Marshal(data)
	if err != nil {
		return nil, err
	}
	req, err := http.NewRequest(
		http.MethodPost, c.endpoint, bytes.NewReader(body),
	)
	if err != nil {
		return nil, err
	}
	req.Header.Add("Content-Type", "text/xml; charset=UTF-8")
	req.Header.Add("SOAPAction", "http://renins.com/IIntegrationService/MakeCalculation")
	respData, err := c.doRequest(req)
	if err != nil {
		return nil, err
	}
	var result envelopeResp
	if err := xml.Unmarshal(respData, &result); err != nil {
		return nil, err
	}
	var kbms []PhysicalPersonKBM
	if resp := result.Body.MakeCalculationResponse.Response; resp != nil {
		for i, row := range resp.Value.CalcKBMResponses {
			if persons[i].Serial != row.PersonDocument.Serial {
				continue
			}
			if persons[i].Number != row.PersonDocument.Number {
				continue
			}
			kbms = append(kbms, PhysicalPersonKBM{
				Serial:   row.PersonDocument.Serial,
				Number:   row.PersonDocument.Number,
				KBM:      row.KBMValue,
				FirstKBM: row.KBMFirstValue,
			})
		}
	}
	return kbms, nil
}

func (c *Client) doRequest(req *http.Request) ([]byte, error) {
	resp, err := c.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer func() {
		if err := resp.Body.Close(); err != nil {
			log.Println("Error:", err)
		}
	}()
	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, err
	}
	if resp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf(
			"wrong response status %q with content %q",
			resp.Status, body,
		)
	}
	return body, nil
}
