package tatneft

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"math"
	"net/http"
	"net/url"
	"strconv"
	"time"

	"github.com/gofrs/uuid"
	"github.com/xuri/excelize/v2"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
)

const (
	// WTF? Tatneft does not support HTTPS.
	Production = "http://lk.tatneft.ru/solar-portal"
)

// Client represents Tatneft client
type Client struct {
	endpoint string
	client   http.Client
	logger   log.Logger
}

// Session represents client session.
type Session struct {
	JSessionID string
	XSRFToken  string
	IDToken    string
}

func (c *Client) Login(login, password string) (*Session, error) {
	csrf, err := uuid.NewV4()
	if err != nil {
		return nil, err
	}
	form := map[string]string{
		"domain":   "DEFAULT",
		"username": login,
		"password": password,
	}
	formData, err := json.Marshal(form)
	if err != nil {
		return nil, err
	}
	req, err := http.NewRequest(
		http.MethodPost, c.endpoint+"/api/authenticate",
		bytes.NewReader(formData),
	)
	if err != nil {
		return nil, err
	}
	req.Header.Set("Accept", "application/json, text/plain, */*")
	req.Header.Set("Content-Type", "application/json")
	req.AddCookie(&http.Cookie{Name: "XSRF-TOKEN", Value: csrf.String()})
	resp, err := c.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer func() {
		if err := resp.Body.Close(); err != nil {
			c.logger.Error("Unable to close body", log.Error(err))
		}
	}()
	if resp.StatusCode != 200 {
		return nil, fmt.Errorf("invalid status code: %d", resp.StatusCode)
	}
	var data struct {
		IDToken string `json:"id_token"`
	}
	if err := json.NewDecoder(resp.Body).Decode(&data); err != nil {
		return nil, err
	}
	session := Session{}
	for _, cookie := range resp.Cookies() {
		if cookie.Value == "" {
			continue
		}
		switch cookie.Name {
		case "JSESSIONID":
			session.JSessionID = cookie.Value
		case "XSRF-TOKEN":
			session.XSRFToken = cookie.Value
		}
	}
	session.IDToken = data.IDToken
	return &session, nil
}

func (c *Client) Logout(s *Session) error {
	req, err := c.newRequest(http.MethodGet, "/api/authLogout", nil, s)
	if err != nil {
		return err
	}
	resp, err := c.client.Do(req)
	if err != nil {
		return err
	}
	defer func() {
		if err := resp.Body.Close(); err != nil {
			c.logger.Error("Unable to close body", log.Error(err))
		}
	}()
	if resp.StatusCode != 200 {
		return fmt.Errorf("invalid status code: %d", resp.StatusCode)
	}
	return nil
}

type Card struct {
	ID     int
	Number string
	Status string
}

func (c *Client) GetCards(s *Session) ([]Card, error) {
	form := struct {
		Pager pager `json:"pager"`
	}{
		Pager: pager{Page: 1, PageSize: 100000},
	}
	formData, err := json.Marshal(form)
	if err != nil {
		return nil, err
	}
	req, err := c.newRequest(
		http.MethodPost, "/api/data/customerCardList/tnp.cardsData/$all",
		bytes.NewReader(formData), s,
	)
	if err != nil {
		return nil, err
	}
	body, err := c.doRequest(req)
	if err != nil {
		return nil, err
	}
	var data struct {
		Data []struct {
			ID         int    `json:"id"`
			CardNumber string `json:"cardNumber"`
			Status     string `json:"status"`
		} `json:"data"`
	}
	if err := json.Unmarshal(body, &data); err != nil {
		return nil, err
	}
	var cards []Card
	for _, card := range data.Data {
		cards = append(cards, Card{
			ID:     card.ID,
			Number: card.CardNumber,
			Status: card.Status,
		})
	}
	return cards, nil
}

type pager struct {
	Page     int `json:"page"`
	PageSize int `json:"pageSize"`
}

type Tx struct {
	DocID         string
	ID            int64
	Time          time.Time
	Card          string
	Total         float64
	DiscountTotal float64
	Type          int
	Category      string
	FuelName      string
	Amount        float64
	Price         float64
	PriceDiscount float64
}

type TxItem struct {
	ID            int64
	Total         float64
	DiscountTotal float64
	Price         float64
	PriceDiscount float64
	Amount        float64
	FuelName      string
}

const eps = 1e-6

func (c *Client) GetCardTxs(
	s *Session, cardNumber string, start, finish time.Time,
) ([]Tx, error) {
	form := struct {
		Pager       pager  `json:"pager"`
		CardNumber  string `json:"cardNumber"`
		TxnDateFrom string `json:"txnDateFrom"`
		TxnDateTo   string `json:"txnDateTo"`
	}{
		Pager:       pager{Page: 1, PageSize: 100000},
		CardNumber:  cardNumber,
		TxnDateFrom: start.Format("2006-01-02"),
		TxnDateTo:   finish.Format("2006-01-02"),
	}
	formData, err := json.Marshal(form)
	if err != nil {
		return nil, err
	}
	req, err := c.newRequest(
		http.MethodPost, "/api/data/cardTxnList/tnp.customerCard/$all",
		bytes.NewReader(formData), s,
	)
	if err != nil {
		return nil, err
	}
	resp, err := c.doRequest(req)
	if err != nil {
		return nil, err
	}
	var data struct {
		Data []struct {
			ID                 int64   `json:"id"`
			TxDate             string  `json:"transactionDate"`
			CardNumber         string  `json:"cardNumber"`
			AmountWithDiscount float64 `json:"amountWithDiscount"`
			Discount           float64 `json:"discount"`
			Amount             float64 `json:"amount"`
			ContractAndDoc     string  `json:"contractAndDoc"`
			TxType             int     `json:"transactionType"`
			RequestCategory    string  `json:"requestCategory"`
		} `json:"data"`
	}
	if err := json.Unmarshal(resp, &data); err != nil {
		return nil, err
	}
	loc, err := time.LoadLocation("Europe/Moscow")
	if err != nil {
		return nil, err
	}
	var txs []Tx
	for _, tx := range data.Data {
		date, err := time.ParseInLocation(
			"2006-01-02T15:04:05", tx.TxDate, loc,
		)
		if err != nil {
			return nil, err
		}
		if math.Abs(tx.AmountWithDiscount+tx.Discount-tx.Amount) > eps {
			return nil, fmt.Errorf(
				"AmountWithDiscount + Discount != Amount",
			)
		}
		txs = append(txs, Tx{
			DocID:         tx.ContractAndDoc,
			ID:            tx.ID,
			Time:          date,
			Card:          tx.CardNumber,
			Total:         tx.AmountWithDiscount,
			DiscountTotal: tx.Discount,
			Type:          tx.TxType,
			Category:      tx.RequestCategory,
		})
	}
	return txs, nil
}

func (c *Client) GetTxItems(s *Session, docID string) ([]TxItem, error) {
	form := struct {
		Pager pager  `json:"pager"`
		DocID string `json:"docId"`
	}{
		Pager: pager{Page: 1, PageSize: 100000},
		DocID: docID,
	}
	formData, err := json.Marshal(form)
	if err != nil {
		return nil, err
	}
	req, err := c.newRequest(
		http.MethodPost, "/api/data/txnLineItemList/tnp.txn/$all",
		bytes.NewReader(formData), s,
	)
	if err != nil {
		return nil, err
	}
	resp, err := c.doRequest(req)
	if err != nil {
		return nil, err
	}
	var data struct {
		Data []struct {
			ID                    int64   `json:"id"`
			GoodsGroupDescription string  `json:"goodsGroupDescription"`
			Quantity              float64 `json:"quantity"`
			Amount                float64 `json:"amount"`
			AmountWithDiscount    float64 `json:"amountWithDiscount"`
			Price                 float64 `json:"price"`
			PriceWithDiscount     float64 `json:"priceWithDiscount"`
		} `json:"data"`
	}
	if err := json.Unmarshal(resp, &data); err != nil {
		return nil, err
	}
	var items []TxItem
	for _, item := range data.Data {
		items = append(items, TxItem{
			ID:            item.ID,
			Total:         item.AmountWithDiscount,
			DiscountTotal: item.Amount - item.AmountWithDiscount,
			Price:         item.PriceWithDiscount,
			PriceDiscount: item.Price - item.PriceWithDiscount,
			Amount:        item.Quantity,
			FuelName:      item.GoodsGroupDescription,
		})
	}
	return items, nil
}

const ReportTemplate = "22020702"

func (c *Client) BuildReport(s *Session, start, finish time.Time) error {
	form := struct {
		DateFrom       string `json:"dateFrom"`
		DateTo         string `json:"dateTo"`
		Format         string `json:"format"`
		ReportParams   string `json:"reportParams"`
		ReportTemplate string `json:"reportTemplate"`
	}{
		DateFrom:       start.Format("2006-01-02"),
		DateTo:         finish.Format("2006-01-02"),
		Format:         "XLSX",
		ReportParams:   ";P_DATE_FROM=Y;P_DATE_TO=Y;P_FMT=;P_ISS_CONTRACT=Y;",
		ReportTemplate: ReportTemplate,
	}
	formData, err := json.Marshal(form)
	if err != nil {
		return err
	}
	req, err := c.newRequest(
		http.MethodPost, "/api/action/reportJob/tnp.reportData/$all",
		bytes.NewReader(formData), s,
	)
	if err != nil {
		return err
	}
	resp, err := c.doRequest(req)
	if err != nil {
		return err
	}
	var data struct {
		Data struct {
			FileName string `json:"fileName"`
		} `json:"data"`
	}
	if err := json.Unmarshal(resp, &data); err != nil {
		return err
	}
	return nil
}

type ReportInfo struct {
	ID        int64
	FileName  string
	StartDate string
	EndDate   string
	Template  string
	Status    string
}

// Finished returns that report construction is finished.
func (r ReportInfo) Finished() bool {
	return r.Status == "FINISHED"
}

func (c *Client) ListReports(
	s *Session, start, finish time.Time,
) ([]ReportInfo, error) {
	form := struct {
		Pager           pager  `json:"pager"`
		DateFrom        string `json:"dateFrom"`
		DateTo          string `json:"dateTo"`
		AgreementNumber string `json:"agreementNumber"`
	}{
		Pager:    pager{Page: 1, PageSize: 10},
		DateFrom: start.Format("2006-01-02"),
		DateTo:   finish.AddDate(0, 0, 2).Format("2006-01-02"),
	}
	formData, err := json.Marshal(form)
	if err != nil {
		return nil, err
	}
	req, err := c.newRequest(
		http.MethodPost, "/api/data/reportList/tnp.reportData/$all",
		bytes.NewReader(formData), s,
	)
	if err != nil {
		return nil, err
	}
	resp, err := c.doRequest(req)
	if err != nil {
		return nil, err
	}
	var data struct {
		Data []struct {
			ID             int64  `json:"id"`
			FileName       string `json:"fileName"`
			StartDate      string `json:"startDate"`
			EndDate        string `json:"endDate"`
			ReportTemplate string `json:"reportTemplate"`
			Status         string `json:"status"`
		} `json:"data"`
	}
	if err := json.Unmarshal(resp, &data); err != nil {
		return nil, err
	}
	var result []ReportInfo
	for _, item := range data.Data {
		result = append(result, ReportInfo{
			ID:        item.ID,
			FileName:  item.FileName,
			StartDate: item.StartDate,
			EndDate:   item.EndDate,
			Template:  item.ReportTemplate,
			Status:    item.Status,
		})
	}
	return result, nil
}

func (c *Client) LoadReport(s *Session, info ReportInfo) ([]Tx, error) {
	req, err := c.newRequest(
		http.MethodGet,
		fmt.Sprintf("/api/report/%d/%s", info.ID, url.PathEscape(info.FileName)),
		nil, s,
	)
	if err != nil {
		return nil, err
	}
	resp, err := c.doRequest(req)
	if err != nil {
		return nil, err
	}
	xlsx, err := excelize.OpenReader(bytes.NewBuffer(resp))
	if err != nil {
		return nil, err
	}
	sheet := xlsx.GetSheetName(xlsx.GetActiveSheetIndex())
	customNumFmt := "[$-409]MM/DD/YYYY"
	style, err := xlsx.NewStyle(&excelize.Style{CustomNumFmt: &customNumFmt})
	if err != nil {
		return nil, err
	}
	if err := xlsx.SetColStyle(sheet, "A", style); err != nil {
		return nil, err
	}
	rows, err := xlsx.GetRows(sheet)
	if err != nil {
		return nil, err
	}
	var txs []Tx
	if len(rows) <= 1 {
		return txs, nil
	}
	for i, row := range rows[1:] {
		dateCell := fmt.Sprintf("A%d", i+2)
		if err := xlsx.SetCellStyle(sheet, dateCell, dateCell, style); err != nil {
			return nil, err
		}
		date, err := xlsx.GetCellValue(sheet, dateCell)
		if err != nil {
			return nil, err
		}
		if date == "" {
			continue
		}
		ts, err := strconv.ParseFloat(date, 64)
		if err != nil {
			return nil, err
		}
		dt, err := excelize.ExcelDateToTime(ts, false)
		if err != nil {
			return nil, err
		}
		total, err := strconv.ParseFloat(row[14], 64)
		if err != nil {
			return nil, err
		}
		totalDiscounted, err := strconv.ParseFloat(row[16], 64)
		if err != nil {
			return nil, err
		}
		amount, err := strconv.ParseFloat(row[11], 64)
		if err != nil {
			return nil, err
		}
		price, err := strconv.ParseFloat(row[13], 64)
		if err != nil {
			return nil, err
		}
		priceDiscounted, err := strconv.ParseFloat(row[15], 64)
		if err != nil {
			return nil, err
		}
		txs = append(txs, Tx{
			Time:          dt.Add(-3 * time.Hour),
			Card:          row[1],
			Total:         totalDiscounted,
			DiscountTotal: total - totalDiscounted,
			FuelName:      row[10],
			Amount:        amount,
			Price:         priceDiscounted,
			PriceDiscount: price - priceDiscounted,
		})
	}
	return txs, nil
}

func (c *Client) newRequest(
	method string, url string, body io.Reader, s *Session,
) (*http.Request, error) {
	req, err := http.NewRequest(method, c.endpoint+url, body)
	if err != nil {
		return nil, err
	}
	req.Header.Add("Content-Type", "application/json;charset=UTF-8")
	req.Header.Add("Accept", "application/json")
	req.Header.Add("X-Requested-With", "XMLHttpRequest")
	if s != nil {
		req.Header.Add("X-XSRF-TOKEN", s.XSRFToken)
		req.Header.Set("Authorization", fmt.Sprintf("Bearer %s", s.IDToken))
		req.AddCookie(&http.Cookie{Name: "JSESSIONID", Value: s.JSessionID})
		req.AddCookie(&http.Cookie{Name: "XSRF-TOKEN", Value: s.XSRFToken})
	}
	return req, nil
}

func (c *Client) doRequest(req *http.Request) ([]byte, error) {
	resp, err := c.client.Do(req)
	if err != nil {
		return nil, err
	}
	defer func() {
		if err := resp.Body.Close(); err != nil {
			c.logger.Error("Unable to close body", log.Error(err))
		}
	}()
	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, err
	}
	if resp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf(
			"wrong response status %q with content %q",
			resp.Status, body,
		)
	}
	return body, nil
}

type ClientOpts func(*Client)

// NewClient creates a new instance of Tatneft client.
//
// You can pass to endpoint "", if you want to use production endpoint, or
// pass "Production" if you want to specify production endpoint explicitly.
func NewClient(endpoint string, opts ...ClientOpts) *Client {
	if endpoint == "" || endpoint == "production" || endpoint == "Production" {
		endpoint = Production
	}
	client := Client{
		endpoint: endpoint,
		client: http.Client{
			Timeout: time.Minute,
			CheckRedirect: func(*http.Request, []*http.Request) error {
				return http.ErrUseLastResponse
			},
		},
	}
	for _, opt := range opts {
		opt(&client)
	}
	if client.logger == nil {
		var err error
		client.logger, err = zap.New(zap.ConsoleConfig(log.InfoLevel))
		if err != nil {
			panic(err)
		}
	}
	return &client
}
