package taxidwh

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"strings"
	"time"

	"a.yandex-team.ru/drive/library/go/auth"
)

const (
	// Production represents production endpoint.
	Production = "http://replication.taxi.yandex.net"
	// Testing represents testing endpoint.
	Testing = "http://replication.taxi.tst.yandex.net"
)

// Client represents Taxi DWH client.
type Client struct {
	endpoint string
	client   http.Client
	auth     auth.Provider
}

// Document represents document with data.
type Document struct {
	ID   string      `json:"id"`
	Data interface{} `json:"data"`
}

// DocumentStatus represents document status.
type DocumentStatus struct {
	ID           string `json:"id"`
	Status       string `json:"status"`
	ErrorMessage string `json:"error_message"`
}

// AddDocuments adds documents to replication queue.
func (c *Client) AddDocuments(
	rule string, docs []Document,
) ([]DocumentStatus, error) {
	var form struct {
		Items []Document `json:"items"`
	}
	form.Items = docs
	jsonForm, err := json.Marshal(form)
	if err != nil {
		return nil, err
	}
	req, err := http.NewRequest(
		http.MethodPost, c.endpoint+fmt.Sprintf("/data/%s", rule),
		bytes.NewReader(jsonForm),
	)
	if err != nil {
		return nil, err
	}
	if err := c.auth.UpdateRequest(req); err != nil {
		return nil, err
	}
	resp, err := c.client.Do(req)
	if err != nil {
		return nil, err
	}
	if resp.StatusCode != 200 {
		data, _ := ioutil.ReadAll(resp.Body)
		return nil, fmt.Errorf(
			"bad status %d: %q", resp.StatusCode, string(data),
		)
	}
	var data struct {
		Items []DocumentStatus `json:"items"`
	}
	if err := json.NewDecoder(resp.Body).Decode(&data); err != nil {
		return nil, err
	}
	return data.Items, nil
}

// NewClient creates a new instance of Taxi DWH client.
func NewClient(endpoint string, auth auth.Provider) *Client {
	switch strings.ToLower(endpoint) {
	case "production", "":
		endpoint = Production
	case "testing":
		endpoint = Testing
	}
	return &Client{
		endpoint: endpoint,
		client:   http.Client{Timeout: time.Minute},
		auth:     auth,
	}
}
