package cron

import (
	"errors"
	"strconv"
	"strings"
)

type bounds struct {
	begin uint
	end   uint
}

var (
	seconds    = bounds{0, 60}
	minutes    = bounds{0, 60}
	hours      = bounds{0, 24}
	dayOfMonth = bounds{1, 32}
	month      = bounds{1, 13}
	dayOfWeek  = bounds{0, 7}
)

func Parse(raw string) (*Schedule, error) {
	fields := strings.Fields(raw)
	if len(fields) > 6 {
		return nil, errors.New("invalid amount of fields")
	}
	for len(fields) < 6 {
		fields = append(fields, "*")
	}
	schedule := Schedule{raw: raw}
	var err error
	schedule.seconds, err = parseField(fields[0], seconds)
	if err != nil {
		return nil, err
	}
	schedule.minutes, err = parseField(fields[1], minutes)
	if err != nil {
		return nil, err
	}
	schedule.hours, err = parseField(fields[2], hours)
	if err != nil {
		return nil, err
	}
	schedule.dayOfMonth, err = parseField(fields[3], dayOfMonth)
	if err != nil {
		return nil, err
	}
	schedule.month, err = parseField(fields[4], month)
	if err != nil {
		return nil, err
	}
	schedule.dayOfWeek, err = parseField(fields[5], dayOfWeek)
	if err != nil {
		return nil, err
	}
	return &schedule, nil
}

func parseField(field string, b bounds) (uint64, error) {
	var bits uint64
	parts := strings.FieldsFunc(field, func(c rune) bool { return c == ',' })
	for _, part := range parts {
		partBits, err := parsePart(part, b)
		if err != nil {
			return 0, err
		}
		bits |= partBits
	}
	return bits, nil
}

func parsePart(part string, b bounds) (uint64, error) {
	rangeAndStep := strings.SplitN(part, "/", 2)
	beginAndEnd := strings.SplitN(rangeAndStep[0], "-", 2)
	step := uint(1)
	if len(rangeAndStep) > 1 {
		var err error
		step, err = parseUint(rangeAndStep[1])
		if err != nil {
			return 0, err
		}
		if step == 0 {
			return 0, errors.New("step should be positive")
		}
	}
	if beginAndEnd[0] == "*" {
		if len(beginAndEnd) > 1 {
			return 0, errors.New("range can not contains '*'")
		}
		return getBits(b.begin, b.end, uint(step)), nil
	} else {
		start, err := parseUint(beginAndEnd[0])
		if err != nil {
			return 0, err
		}
		if start < b.begin || start >= b.end {
			return 0, errors.New("invalid start of range")
		}
		finish := start
		if len(beginAndEnd) > 1 {
			var err error
			finish, err = parseUint(beginAndEnd[1])
			if err != nil {
				return 0, err
			}
			if finish < start || finish < b.begin || finish >= b.end {
				return 0, errors.New("invalid finish of range")
			}
		}
		return getBits(start, finish+1, step), nil
	}
}

func getBits(begin, end, step uint) uint64 {
	var bits uint64
	for i := begin; i < end; i += step {
		bits |= 1 << i
	}
	return bits
}

func parseUint(s string) (uint, error) {
	v, err := strconv.Atoi(s)
	if err != nil {
		return 0, err
	}
	if v < 0 {
		return 0, errors.New("number should be non negative")
	}
	return uint(v), nil
}
