package cron

import (
	"time"
)

// Schedule represents cron schedule.
type Schedule struct {
	seconds    uint64
	minutes    uint64
	hours      uint64
	dayOfMonth uint64
	month      uint64
	dayOfWeek  uint64
	raw        string
}

// Next returns first time that exists in schedule strongly greater than
// specified time or zero value if there is no such occurrences.
func (s Schedule) Next(t time.Time) time.Time {
	t = t.Add(time.Second)
	limit := t.AddDate(50, 0, 0)
loop:
	for {
		for !s.monthMatch(t) && t.Before(limit) {
			t = time.Date(t.Year(), t.Month(), 1, 0, 0, 0, 0, t.Location())
			t = t.AddDate(0, 1, 0)
		}
		for !s.dayMatch(t) && t.Before(limit) {
			t = time.Date(t.Year(), t.Month(), t.Day(), 0, 0, 0, 0, t.Location())
			t = t.AddDate(0, 0, 1)
			if t.Day() == 1 {
				continue loop
			}
		}
		for !s.hourMatch(t) && t.Before(limit) {
			t = t.Truncate(time.Hour).Add(time.Hour)
			if t.Hour() == 0 {
				continue loop
			}
		}
		for !s.minuteMatch(t) && t.Before(limit) {
			t = t.Truncate(time.Minute).Add(time.Minute)
			if t.Minute() == 0 {
				continue loop
			}
		}
		for !s.secondMatch(t) && t.Before(limit) {
			t = t.Truncate(time.Second).Add(time.Second)
			if t.Second() == 0 {
				continue loop
			}
		}
		if t.Before(limit) {
			return t
		}
		return time.Time{}
	}
}

func (s Schedule) String() string {
	return s.raw
}

func (s Schedule) MarshalText() ([]byte, error) {
	return []byte(s.raw), nil
}

func (s *Schedule) UnmarshalText(bytes []byte) error {
	schedule, err := Parse(string(bytes))
	if err != nil {
		return err
	}
	*s = *schedule
	return nil
}

func (s Schedule) monthMatch(t time.Time) bool {
	return 1<<uint(t.Month())&s.month > 0
}

func (s Schedule) dayMatch(t time.Time) bool {
	return 1<<uint(t.Day())&s.dayOfMonth > 0 &&
		1<<uint(t.Weekday())&s.dayOfWeek > 0
}

func (s Schedule) hourMatch(t time.Time) bool {
	return 1<<uint(t.Hour())&s.hours > 0
}

func (s Schedule) minuteMatch(t time.Time) bool {
	return 1<<uint(t.Minute())&s.minutes > 0
}

func (s Schedule) secondMatch(t time.Time) bool {
	return 1<<uint(t.Second())&s.seconds > 0
}
