package echolog

import (
	"fmt"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/log"
)

// Logger returns middleware for echo.
func Logger(logger log.Logger) echo.MiddlewareFunc {
	return func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) error {
			req := c.Request()
			resp := c.Response()
			start := time.Now()
			errResp := next(c)
			if errResp != nil {
				c.Error(errResp)
			}
			stop := time.Now()
			requestID := req.Header.Get(echo.HeaderXRequestID)
			if requestID == "" {
				requestID = resp.Header().Get(echo.HeaderXRequestID)
			}
			fields := []log.Field{
				log.Time("time", stop),
				log.Duration("latency", stop.Sub(start)),
				log.String("remote_ip", c.RealIP()),
				log.String("host", req.Host),
				log.String("method", req.Method),
				log.String("uri", req.RequestURI),
				log.Int("status", resp.Status),
				log.Int64("size", resp.Size),
				log.String("user_agent", req.UserAgent()),
				log.String("request_id", requestID),
				log.Error(errResp),
			}
			message := fmt.Sprintf("%s %s", req.Method, req.URL.Path)
			switch {
			case resp.Status >= 500:
				logger.Error(message, fields...)
			case resp.Status >= 400:
				logger.Warn(message, fields...)
			case resp.Status >= 300:
				logger.Info(message, fields...)
			default:
				logger.Info(message, fields...)
			}
			return nil
		}
	}
}
