package exprs

import (
	"fmt"
	"math"
	"reflect"
	"strconv"
)

type Scope interface {
	Value(name string) interface{}
	Call(name string, args ...interface{}) interface{}
}

type Operator interface {
	Exec(s Scope) interface{}
}

type logicalOpKind int

const (
	orLogicalOp logicalOpKind = iota
	andLogicalOp
)

type logicalOp struct {
	kind logicalOpKind
	ops  []Operator
}

func (o logicalOp) Exec(s Scope) interface{} {
	switch o.kind {
	case orLogicalOp:
		for _, op := range o.ops {
			if v := op.Exec(s); !reflect.ValueOf(v).IsZero() {
				return true
			}
		}
		return false
	case andLogicalOp:
		for _, op := range o.ops {
			if v := op.Exec(s); reflect.ValueOf(v).IsZero() {
				return false
			}
		}
		return true
	}
	panic(fmt.Errorf("unsupported logical operator: %d", o.kind))
}

type unaryOpKind int

const (
	notOp unaryOpKind = iota
	isNilOp
	isNotNilOp
)

type unaryOp struct {
	kind unaryOpKind
	op   Operator
}

func (o unaryOp) Exec(s Scope) interface{} {
	switch x := o.op.Exec(s); o.kind {
	case notOp:
		return reflect.ValueOf(x).IsZero()
	case isNilOp:
		v := reflect.ValueOf(x)
		return v.Kind() == reflect.Ptr && v.IsNil()
	case isNotNilOp:
		v := reflect.ValueOf(x)
		return v.Kind() != reflect.Ptr || !v.IsNil()
	}
	panic(fmt.Errorf("unsupported unary operator: %d", o.kind))
}

type binaryOpKind int

const (
	equalOp binaryOpKind = iota
	notEqualOp
	lessOp
	greaterOp
	lessEqualOp
	greaterEqualOp
)

type binaryOp struct {
	kind     binaryOpKind
	lhs, rhs Operator
}

func (o binaryOp) Exec(s Scope) interface{} {
	switch cmp := compare(o.lhs.Exec(s), o.rhs.Exec(s)); o.kind {
	case equalOp:
		return cmp == 0
	case notEqualOp:
		return cmp != 0
	case lessOp:
		return cmp == -1
	case greaterOp:
		return cmp == 1
	case lessEqualOp:
		return cmp == -1 || cmp == 0
	case greaterEqualOp:
		return cmp == 1 || cmp == 0
	}
	panic(fmt.Errorf("unsupported compare operator: %d", o.kind))
}

type valueOp struct {
	name string
}

func (o valueOp) Exec(s Scope) interface{} {
	return s.Value(o.name)
}

type callOp struct {
	name string
	args []Operator
}

func (o callOp) Exec(s Scope) interface{} {
	var args []interface{}
	for _, arg := range o.args {
		args = append(args, arg.Exec(s))
	}
	return s.Call(o.name, args...)
}

type constValueOp struct {
	value interface{}
}

func (o constValueOp) Exec(s Scope) interface{} {
	return o.value
}

func compare(lhs, rhs interface{}) int {
	switch v := reflect.ValueOf(lhs); v.Kind() {
	case reflect.Bool:
		return compareBool(v.Bool(), rhs)
	case reflect.Int, reflect.Int8, reflect.Int16,
		reflect.Int32, reflect.Int64:
		return compareInt(v.Int(), rhs)
	case reflect.Uint, reflect.Uint8, reflect.Uint16,
		reflect.Uint32, reflect.Uint64:
		return compareUint(v.Uint(), rhs)
	case reflect.Float32, reflect.Float64:
		return compareFloat(v.Float(), rhs)
	case reflect.String:
		return compareString(v.String(), rhs)
	}
	return 2
}

func compareBool(lhs bool, rhs interface{}) int {
	switch v := reflect.ValueOf(rhs); v.Kind() {
	case reflect.Bool:
		if lhs == v.Bool() {
			return 0
		}
	case reflect.Int, reflect.Int8, reflect.Int16,
		reflect.Int32, reflect.Int64:
		if lhs == (v.Int() != 0) {
			return 0
		}
	case reflect.Uint, reflect.Uint8, reflect.Uint16,
		reflect.Uint32, reflect.Uint64:
		if lhs == (v.Uint() != 0) {
			return 0
		}
	case reflect.Float32, reflect.Float64:
		if lhs == (v.Float() != 0) {
			return 0
		}
	case reflect.String:
		if lhs == (v.Len() != 0) {
			return 0
		}
	}
	return 2
}

func compareInt(lhs int64, rhs interface{}) int {
	switch v := reflect.ValueOf(rhs); v.Kind() {
	case reflect.Bool:
		if (lhs != 0) == v.Bool() {
			return 0
		}
	case reflect.Int, reflect.Int8, reflect.Int16,
		reflect.Int32, reflect.Int64:
		if lhs < v.Int() {
			return -1
		} else if lhs > v.Int() {
			return 1
		} else {
			return 0
		}
	case reflect.Uint, reflect.Uint8, reflect.Uint16,
		reflect.Uint32, reflect.Uint64:
		if lhs < int64(v.Uint()) {
			return -1
		} else if lhs > int64(v.Uint()) {
			return 1
		} else {
			return 0
		}
	case reflect.Float32, reflect.Float64:
		if float64(lhs) < v.Float() {
			return -1
		} else if float64(lhs) > v.Float() {
			return 1
		} else {
			return 0
		}
	}
	return 2
}

func compareUint(lhs uint64, rhs interface{}) int {
	switch v := reflect.ValueOf(rhs); v.Kind() {
	case reflect.Bool:
		if (lhs != 0) == v.Bool() {
			return 0
		}
	case reflect.Int, reflect.Int8, reflect.Int16,
		reflect.Int32, reflect.Int64:
		if int64(lhs) < v.Int() {
			return -1
		} else if int64(lhs) > v.Int() {
			return 1
		} else {
			return 0
		}
	case reflect.Uint, reflect.Uint8, reflect.Uint16,
		reflect.Uint32, reflect.Uint64:
		if lhs < v.Uint() {
			return -1
		} else if lhs > v.Uint() {
			return 1
		} else {
			return 0
		}
	case reflect.Float32, reflect.Float64:
		if float64(lhs) < v.Float() {
			return -1
		} else if float64(lhs) > v.Float() {
			return 1
		} else {
			return 0
		}
	}
	return 2
}

const eps = 1e-12

func compareFloats(lhs, rhs float64) int {
	x := math.Abs(lhs - rhs)
	y := math.Abs(lhs)
	if v := math.Abs(rhs); v > y {
		y = v
	}
	if x <= y*eps {
		return 0
	} else if lhs < rhs {
		return -1
	} else {
		return 1
	}
}

func compareFloat(lhs float64, rhs interface{}) int {
	switch v := reflect.ValueOf(rhs); v.Kind() {
	case reflect.Bool:
		if (lhs != 0) == v.Bool() {
			return 0
		}
	case reflect.Int, reflect.Int8, reflect.Int16,
		reflect.Int32, reflect.Int64:
		if lhs < float64(v.Int()) {
			return -1
		} else if lhs > float64(v.Int()) {
			return 1
		} else {
			return 0
		}
	case reflect.Uint, reflect.Uint8, reflect.Uint16,
		reflect.Uint32, reflect.Uint64:
		if lhs < float64(v.Uint()) {
			return -1
		} else if lhs > float64(v.Uint()) {
			return 1
		} else {
			return 0
		}
	case reflect.Float32, reflect.Float64:
		if lhs < v.Float() {
			return -1
		} else if lhs > v.Float() {
			return 1
		} else {
			return 0
		}
	case reflect.String:
		x, err := strconv.ParseFloat(v.String(), 64)
		if err != nil {
			return 2
		}
		return compareFloats(lhs, x)
	}
	return 2
}

func compareString(lhs string, rhs interface{}) int {
	switch v := reflect.ValueOf(rhs); v.Kind() {
	case reflect.Bool:
		if (len(lhs) != 0) == v.Bool() {
			return 0
		}
	case reflect.Float32, reflect.Float64:
		x, err := strconv.ParseFloat(lhs, 64)
		if err != nil {
			return 2
		}
		return compareFloats(x, v.Float())
	case reflect.String:
		if lhs == v.String() {
			return 0
		}
	}
	return 2
}
