package gocfg

import (
	"bytes"
	"encoding/json"
	"io/ioutil"
	"os"
	"path/filepath"
	"strings"
	"text/template"
)

var configFuncs = template.FuncMap{
	"json": func(value interface{}) (string, error) {
		data, err := json.Marshal(value)
		return string(data), err
	},
	"file": func(name string) (string, error) {
		bytes, err := ioutil.ReadFile(name)
		if err != nil {
			return "", err
		}
		return strings.TrimRight(string(bytes), "\r\n"), nil
	},
	"env": os.Getenv,
}

// ParseFile loads config from file.
func ParseFile(path string, cfg interface{}) error {
	name := filepath.Base(path)
	tmpl, err := template.New(name).Funcs(configFuncs).ParseFiles(path)
	if err != nil {
		return err
	}
	var buffer bytes.Buffer
	if err := tmpl.Execute(&buffer, nil); err != nil {
		return err
	}
	return json.NewDecoder(&buffer).Decode(cfg)
}
