package gosql

import (
	"fmt"
	"strings"
)

// Builder represents SQL query builder.
type Builder interface {
	Insert(table string) InsertQuery
	Select(table string) SelectQuery
	Update(table string) UpdateQuery
	Delete(table string) DeleteQuery
}

// Query represents builder for generic SQL query.
type Query interface {
	fmt.Stringer
	Build() (string, []interface{})
}

// InsertQuery represents builder for SQL INSERT query.
type InsertQuery interface {
	Query
	Names(names ...string) InsertQuery
	Values(values ...interface{}) InsertQuery
	Returning(names ...string) InsertQuery
}

// SelectQuery represents builder for SQL SELECT query.
type SelectQuery interface {
	Query
	Where(where BoolExpr) SelectQuery
	Names(names ...string) SelectQuery
	ForUpdate(flag bool) SelectQuery
	OrderBy(names ...interface{}) SelectQuery
	Limit(limit int) SelectQuery
}

// UpdateQuery represents builder for SQL UPDATE query.
type UpdateQuery interface {
	Query
	Where(where BoolExpr) UpdateQuery
	Names(names ...string) UpdateQuery
	Values(values ...interface{}) UpdateQuery
}

// DeleteQuery represents builder for SQL DELETE query.
type DeleteQuery interface {
	Query
	Where(where BoolExpr) DeleteQuery
}

// NewBuilder creates new instance of SQL query builder.
func NewBuilder(driver Driver) Builder {
	return &builder{driver: driver}
}

type builder struct {
	driver Driver
}

func (b builder) Insert(table string) InsertQuery {
	return insertQuery{builder: &b, table: table}
}

func (b builder) Select(table string) SelectQuery {
	return selectQuery{builder: &b, table: table}
}

func (b builder) Update(table string) UpdateQuery {
	return updateQuery{builder: &b, table: table}
}

func (b builder) Delete(table string) DeleteQuery {
	return deleteQuery{builder: &b, table: table}
}

func (b builder) buildName(name string) string {
	return fmt.Sprintf("%q", name)
}

type insertQuery struct {
	builder   *builder
	table     string
	names     []string
	values    []interface{}
	returning []string
}

func (q insertQuery) Names(names ...string) InsertQuery {
	q.names = names
	return q
}

func (q insertQuery) Values(values ...interface{}) InsertQuery {
	q.values = values
	return q
}

func (q insertQuery) Returning(names ...string) InsertQuery {
	q.returning = names
	return q
}

func (q insertQuery) Build() (string, []interface{}) {
	var query strings.Builder
	var opts []interface{}
	query.WriteString("INSERT INTO ")
	query.WriteString(q.builder.buildName(q.table))
	query.WriteString(" (")
	q.buildNames(&query, q.names)
	query.WriteString(") VALUES (")
	q.buildValues(&query, &opts)
	query.WriteRune(')')
	if len(q.returning) > 0 {
		query.WriteString(" RETURNING ")
		q.buildNames(&query, q.returning)
	}
	return query.String(), opts
}

func (q insertQuery) buildNames(query *strings.Builder, names []string) {
	for i, name := range names {
		if i > 0 {
			query.WriteString(", ")
		}
		query.WriteString(q.builder.buildName(name))
	}
}

func (q insertQuery) buildValues(
	query *strings.Builder, values *[]interface{},
) {
	if nLen, vLen := len(q.names), len(q.values); nLen != vLen {
		panic(fmt.Errorf("got %d names and %d values", nLen, vLen))
	}
	for i, value := range q.values {
		if i > 0 {
			query.WriteString(", ")
		}
		*values = append(*values, value)
		query.WriteString(fmt.Sprintf("$%d", len(*values)))
	}
}

func (q insertQuery) String() string {
	query, _ := q.Build()
	return query
}

type selectQuery struct {
	builder   *builder
	table     string
	where     BoolExpr
	names     []string
	orderBy   []interface{}
	limit     int
	forUpdate bool
}

func (q selectQuery) Where(where BoolExpr) SelectQuery {
	q.where = where
	return q
}

func (q selectQuery) Names(names ...string) SelectQuery {
	q.names = names
	return q
}

func (q selectQuery) ForUpdate(flag bool) SelectQuery {
	q.forUpdate = flag
	return q
}

func (q selectQuery) OrderBy(names ...interface{}) SelectQuery {
	q.orderBy = names
	return q
}

func (q selectQuery) Limit(limit int) SelectQuery {
	q.limit = limit
	return q
}

func (q selectQuery) Build() (string, []interface{}) {
	var query strings.Builder
	var opts []interface{}
	query.WriteString("SELECT ")
	q.buildNames(&query, &opts)
	query.WriteString(" FROM ")
	query.WriteString(q.builder.buildName(q.table))
	query.WriteString(" WHERE ")
	q.buildWhere(&query, &opts)
	if len(q.orderBy) > 0 {
		query.WriteString(" ORDER BY ")
		q.buildOrderBy(&query, &opts)
	}
	if q.limit > 0 {
		query.WriteString(fmt.Sprintf(" LIMIT %d", q.limit))
	}
	if q.forUpdate {
		query.WriteString(" FOR UPDATE")
	}
	return query.String(), opts
}

func (q selectQuery) buildNames(
	query *strings.Builder, values *[]interface{},
) {
	if len(q.names) == 0 {
		query.WriteRune('*')
		return
	}
	for i, name := range q.names {
		if i > 0 {
			query.WriteString(", ")
		}
		query.WriteString(q.builder.buildName(name))
	}
}

func (q selectQuery) buildWhere(
	query *strings.Builder, values *[]interface{},
) {
	if q.where == nil {
		query.WriteString("TRUE")
		return
	}
	q.where.build(q.builder, query, values)
}

func (q selectQuery) buildOrderBy(
	query *strings.Builder, values *[]interface{},
) {
	for i, name := range q.orderBy {
		if i > 0 {
			query.WriteString(", ")
		}
		switch v := name.(type) {
		case string:
			Column(v).build(q.builder, query, values)
		case exprBuilder:
			v.build(q.builder, query, values)
		default:
			panic(fmt.Sprintf("type %t does not supported", v))
		}
	}
}

func (q selectQuery) String() string {
	query, _ := q.Build()
	return query
}

type updateQuery struct {
	builder *builder
	table   string
	where   BoolExpr
	names   []string
	values  []interface{}
}

func (q updateQuery) Where(where BoolExpr) UpdateQuery {
	q.where = where
	return q
}

func (q updateQuery) Names(names ...string) UpdateQuery {
	q.names = names
	return q
}

func (q updateQuery) Values(values ...interface{}) UpdateQuery {
	q.values = values
	return q
}

func (q updateQuery) Build() (string, []interface{}) {
	var query strings.Builder
	var opts []interface{}
	query.WriteString("UPDATE ")
	query.WriteString(q.builder.buildName(q.table))
	query.WriteString(" SET ")
	q.buildNameValues(&query, &opts)
	query.WriteString(" WHERE ")
	q.buildWhere(&query, &opts)
	return query.String(), opts
}

func (q updateQuery) buildNameValues(
	query *strings.Builder, values *[]interface{},
) {
	if nLen, vLen := len(q.names), len(q.values); nLen != vLen {
		panic(fmt.Errorf("got %d names and %d values", nLen, vLen))
	}
	for i, name := range q.names {
		if i > 0 {
			query.WriteString(", ")
		}
		query.WriteString(q.builder.buildName(name))
		*values = append(*values, q.values[i])
		query.WriteString(fmt.Sprintf(" = $%d", len(*values)))
	}
}

func (q updateQuery) buildWhere(
	query *strings.Builder, values *[]interface{},
) {
	if q.where == nil {
		query.WriteString("TRUE")
		return
	}
	q.where.build(q.builder, query, values)
}

func (q updateQuery) String() string {
	query, _ := q.Build()
	return query
}

type deleteQuery struct {
	builder *builder
	table   string
	where   BoolExpr
}

func (q deleteQuery) Where(where BoolExpr) DeleteQuery {
	q.where = where
	return q
}

func (q deleteQuery) Build() (string, []interface{}) {
	var query strings.Builder
	var opts []interface{}
	query.WriteString("DELETE FROM ")
	query.WriteString(q.builder.buildName(q.table))
	query.WriteString(" WHERE ")
	q.buildWhere(&query, &opts)
	return query.String(), opts
}

func (q deleteQuery) buildWhere(
	query *strings.Builder, values *[]interface{},
) {
	if q.where == nil {
		query.WriteString("TRUE")
		return
	}
	q.where.build(q.builder, query, values)
}

func (q deleteQuery) String() string {
	query, _ := q.Build()
	return query
}
