package gosql

import (
	"encoding/json"
	"fmt"

	"a.yandex-team.ru/drive/library/go/secret"
)

type Driver string

const (
	PostgresDriver   Driver = "postgres"
	ClickHouseDriver Driver = "clickhouse"
	SQLiteDriver     Driver = "sqlite"
	RawDriver        Driver = "raw"
)

type Config struct {
	// Driver contains driver name.
	Driver Driver `json:"driver"`
	// Options contains driver options.
	Options interface{} `json:"options"`
}

type PostgresOptions struct {
	// Host contains host.
	Host string `json:"host"`
	// Port contains port.
	Port int `json:"port"`
	// Name contains database name.
	Name string `json:"name"`
	// User contains user.
	User string `json:"user"`
	// Password contains password.
	Password secret.Secret `json:"password"`
	// SSLMode contains ssl mode.
	SSLMode string `json:"ssl_mode"`
	// Hosts contains list of replicated hosts.
	Hosts []string `json:"hosts"`
	// TargetSessionAttrs contains attributes.
	TargetSessionAttrs string `json:"target_session_attrs"`
	// StatementTimeout.
	StatementTimeout int64 `json:"statement_timeout"`
	// IdleInTransactionSessionTimeout.
	IdleInTransactionSessionTimeout int64 `json:"idle_in_transaction_session_timeout"`
	// LockTimeout.
	LockTimeout int64 `json:"lock_timeout"`
}

type ClickHouseOptions struct {
	// Hosts contains list of replicated hosts.
	Hosts []string `json:"hosts"`
	// Name contains database name.
	Name string `json:"name"`
	// User contains user.
	User string `json:"user"`
	// Password contains password.
	Password secret.Secret `json:"password"`
}

type SQLiteOptions struct {
	// Path contains path to sqlite file.
	Path string `json:"path"`
	// Mode contains SQLite mode.
	//
	// Valid values: "ro", "rw", "rwc", "memory".
	Mode string `json:"mode"`
}

// RawOptions represents raw connection string without any preprocessing.
type RawOptions struct {
	Driver string        `json:"driver"`
	String secret.Secret `json:"string"`
}

func (d *Config) UnmarshalJSON(data []byte) error {
	var value struct {
		Driver  Driver          `json:"driver"`
		Options json.RawMessage `json:"options"`
	}
	if err := json.Unmarshal(data, &value); err != nil {
		return err
	}
	switch value.Driver {
	case PostgresDriver:
		var options PostgresOptions
		if err := json.Unmarshal(value.Options, &options); err != nil {
			return err
		}
		d.Driver, d.Options = value.Driver, options
	case ClickHouseDriver:
		var options ClickHouseOptions
		if err := json.Unmarshal(value.Options, &options); err != nil {
			return err
		}
		d.Driver, d.Options = value.Driver, options
	case SQLiteDriver:
		var options SQLiteOptions
		if err := json.Unmarshal(value.Options, &options); err != nil {
			return err
		}
		d.Driver, d.Options = value.Driver, options
	case RawDriver:
		var options RawOptions
		if err := json.Unmarshal(value.Options, &options); err != nil {
			return err
		}
		d.Driver, d.Options = value.Driver, options
	default:
		return fmt.Errorf("unsupported driver %q", value.Driver)
	}
	return nil
}
