package gosql

import (
	"context"
	"database/sql"
	"fmt"
	"net/url"
	"strings"

	"github.com/jackc/pgx/v4"
	"github.com/jackc/pgx/v4/stdlib"
)

// DB represents connection to database.
type DB struct {
	*sql.DB
	Builder
	// RO contains read-only connection (read-write also).
	RO *sql.DB
	// Driver contains name of connection driver.
	Driver Driver
}

// BeginTx starts new transaction.
func (db *DB) BeginTx(ctx context.Context, opts *sql.TxOptions) (*sql.Tx, error) {
	if opts != nil && opts.ReadOnly {
		return db.RO.BeginTx(ctx, opts)
	}
	return db.DB.BeginTx(ctx, opts)
}

// Stats returns total stats for database.
func (db *DB) Stats() sql.DBStats {
	stats := db.DB.Stats()
	roStats := db.RO.Stats()
	stats.MaxOpenConnections += roStats.MaxOpenConnections
	stats.OpenConnections += roStats.OpenConnections
	stats.InUse += roStats.InUse
	stats.Idle += roStats.Idle
	stats.WaitCount += roStats.WaitCount
	stats.WaitDuration += roStats.WaitDuration
	stats.MaxIdleClosed += roStats.MaxIdleClosed
	stats.MaxIdleTimeClosed += roStats.MaxIdleTimeClosed
	stats.MaxLifetimeClosed += roStats.MaxLifetimeClosed
	return stats
}

// NewDB creates new instance of database connection.
func NewDB(cfg Config) (*DB, error) {
	conn, err := newDB(cfg)
	if err != nil {
		return nil, err
	}
	roConn, err := newRoDB(conn, cfg)
	if err != nil {
		return nil, err
	}
	driver := cfg.Driver
	if driver == "raw" {
		driver = PostgresDriver
	}
	return &DB{
		DB:      conn,
		RO:      roConn,
		Builder: NewBuilder(cfg.Driver),
		Driver:  driver,
	}, nil
}

func newDB(cfg Config) (*sql.DB, error) {
	switch cfg.Driver {
	case PostgresDriver:
		opts := cfg.Options.(PostgresOptions)
		sslMode := opts.SSLMode
		if sslMode == "" {
			sslMode = "prefer"
		}
		var dbHost, dbPort string
		if len(opts.Hosts) > 0 {
			for _, host := range opts.Hosts {
				parts := strings.SplitN(host, ":", 2)
				if len(parts) == 0 {
					continue
				}
				if len(dbHost) > 0 {
					dbHost += ","
					dbPort += ","
				}
				dbHost += parts[0]
				if len(parts) > 1 {
					dbPort += parts[1]
				} else {
					dbPort += "5432"
				}
			}
		} else {
			dbHost, dbPort = opts.Host, fmt.Sprint(opts.Port)
		}
		connString := fmt.Sprintf(
			"host=%s port=%s dbname=%s user=%s password=%s"+
				" sslmode=%s statement_cache_mode=describe",
			dbHost, dbPort, opts.Name, opts.User,
			opts.Password.Secret(), sslMode,
		)
		if opts.TargetSessionAttrs != "" {
			connString += fmt.Sprintf(
				" target_session_attrs=%s",
				opts.TargetSessionAttrs,
			)
		}
		if statementTimeout := opts.StatementTimeout; statementTimeout >= 0 {
			if statementTimeout == 0 {
				statementTimeout = 120 * 1000 // 2 min.
			}
			connString += fmt.Sprintf(" statement_timeout=%d", statementTimeout)
		}
		idleTransactionTimeout := opts.IdleInTransactionSessionTimeout
		if idleTransactionTimeout >= 0 {
			if idleTransactionTimeout == 0 {
				idleTransactionTimeout = 120 * 1000 // 2 min.
			}
			connString += fmt.Sprintf(
				" idle_in_transaction_session_timeout=%d",
				idleTransactionTimeout,
			)
		}
		if lockTimeout := opts.LockTimeout; lockTimeout >= 0 {
			if lockTimeout == 0 {
				lockTimeout = 120 * 1000 // 2 min.
			}
			connString += fmt.Sprintf(" lock_timeout=%d", lockTimeout)
		}
		connConfig, err := pgx.ParseConfig(connString)
		if err != nil {
			return nil, err
		}
		return stdlib.OpenDB(
			*connConfig,
			stdlib.OptionBeforeConnect(stdlib.RandomizeHostOrderFunc),
		), nil
	case ClickHouseDriver:
		opts := cfg.Options.(ClickHouseOptions)
		params := url.Values{}
		params.Set("username", opts.User)
		params.Set("password", opts.Password.Secret())
		params.Set("database", opts.Name)
		params.Set("secure", "true")
		if len(opts.Hosts) > 1 {
			params.Set("alt_hosts", strings.Join(opts.Hosts[1:], ","))
		}
		return sql.Open("clickhouse", fmt.Sprintf(
			"tcp://%s?%s", opts.Hosts[0], params.Encode(),
		))
	case SQLiteDriver:
		opts := cfg.Options.(SQLiteOptions)
		params := url.Values{}
		if opts.Mode != "" {
			params.Set("mode", opts.Mode)
		}
		if opts.Mode == "memory" || opts.Path == ":memory:" {
			params.Set("cache", "shared")
		}
		return sql.Open("sqlite3", fmt.Sprintf(
			"file:%s?%s", opts.Path, params.Encode(),
		))
	case RawDriver:
		opts := cfg.Options.(RawOptions)
		return sql.Open(opts.Driver, opts.String.Secret())
	default:
		return nil, fmt.Errorf("unsupported driver %q", cfg.Driver)
	}
}

func newRoDB(db *sql.DB, cfg Config) (*sql.DB, error) {
	switch cfg.Driver {
	case PostgresDriver:
		opts := cfg.Options.(PostgresOptions)
		if opts.TargetSessionAttrs != "read-write" {
			return db, nil
		}
		opts.TargetSessionAttrs = "any"
		return newDB(Config{Driver: PostgresDriver, Options: opts})
	default:
		return db, nil
	}
}

// Runner represents object that can run queries.
type Runner interface {
	Exec(string, ...any) (sql.Result, error)
	ExecContext(context.Context, string, ...any) (sql.Result, error)
	Query(string, ...any) (*sql.Rows, error)
	QueryContext(context.Context, string, ...any) (*sql.Rows, error)
	QueryRow(string, ...any) *sql.Row
	QueryRowContext(context.Context, string, ...any) *sql.Row
}

// TxBeginner represents object that can begin transaction.
type TxBeginner interface {
	Begin() (*sql.Tx, error)
	BeginTx(ctx context.Context, opts *sql.TxOptions) (*sql.Tx, error)
}
