package gosql

import (
	"fmt"
	"strings"
)

type exprBuilder interface {
	build(builder *builder, query *strings.Builder, values *[]interface{})
}

type BoolExpr interface {
	exprBuilder
	And(BoolExpr) BoolExpr
	Or(BoolExpr) BoolExpr
}

type Value interface {
	exprBuilder
	Equal(interface{}) BoolExpr
	NotEqual(interface{}) BoolExpr
	Less(interface{}) BoolExpr
	Greater(interface{}) BoolExpr
	LessEqual(interface{}) BoolExpr
	GreaterEqual(interface{}) BoolExpr
}

type SortAsc string

func (c SortAsc) build(
	builder *builder, query *strings.Builder, _ *[]interface{},
) {
	query.WriteString(builder.buildName(string(c)))
	query.WriteString(" ASC")
}

type SortDesc string

func (c SortDesc) build(
	builder *builder, query *strings.Builder, _ *[]interface{},
) {
	query.WriteString(builder.buildName(string(c)))
	query.WriteString(" DESC")
}

type Column string

func (c Column) Equal(o interface{}) BoolExpr {
	if _, ok := o.(Value); !ok {
		o = value{value: o}
	}
	return cmp{kind: eqCmp, lhs: c, rhs: o.(Value)}
}

func (c Column) NotEqual(o interface{}) BoolExpr {
	if _, ok := o.(Value); !ok {
		o = value{value: o}
	}
	return cmp{kind: notEqCmp, lhs: c, rhs: o.(Value)}
}

func (c Column) Less(o interface{}) BoolExpr {
	if _, ok := o.(Value); !ok {
		o = value{value: o}
	}
	return cmp{kind: lessCmp, lhs: c, rhs: o.(Value)}
}

func (c Column) Greater(o interface{}) BoolExpr {
	if _, ok := o.(Value); !ok {
		o = value{value: o}
	}
	return cmp{kind: greaterCmp, lhs: c, rhs: o.(Value)}
}

func (c Column) LessEqual(o interface{}) BoolExpr {
	if _, ok := o.(Value); !ok {
		o = value{value: o}
	}
	return cmp{kind: lessEqualCmp, lhs: c, rhs: o.(Value)}
}

func (c Column) GreaterEqual(o interface{}) BoolExpr {
	if _, ok := o.(Value); !ok {
		o = value{value: o}
	}
	return cmp{kind: greaterEqualCmp, lhs: c, rhs: o.(Value)}
}

func (c Column) build(
	builder *builder, query *strings.Builder, _ *[]interface{},
) {
	query.WriteString(builder.buildName(string(c)))
}

type value struct {
	value interface{}
}

func (v value) Equal(o interface{}) BoolExpr {
	if _, ok := o.(Value); !ok {
		o = value{value: o}
	}
	return cmp{kind: eqCmp, lhs: v, rhs: o.(Value)}
}

func (v value) NotEqual(o interface{}) BoolExpr {
	if _, ok := o.(Value); !ok {
		o = value{value: o}
	}
	return cmp{kind: notEqCmp, lhs: v, rhs: o.(Value)}
}

func (v value) Less(o interface{}) BoolExpr {
	if _, ok := o.(Value); !ok {
		o = value{value: o}
	}
	return cmp{kind: lessCmp, lhs: v, rhs: o.(Value)}
}

func (v value) Greater(o interface{}) BoolExpr {
	if _, ok := o.(Value); !ok {
		o = value{value: o}
	}
	return cmp{kind: greaterCmp, lhs: v, rhs: o.(Value)}
}

func (v value) LessEqual(o interface{}) BoolExpr {
	if _, ok := o.(Value); !ok {
		o = value{value: o}
	}
	return cmp{kind: lessEqualCmp, lhs: v, rhs: o.(Value)}
}

func (v value) GreaterEqual(o interface{}) BoolExpr {
	if _, ok := o.(Value); !ok {
		o = value{value: o}
	}
	return cmp{kind: greaterEqualCmp, lhs: v, rhs: o.(Value)}
}

func (v value) build(
	builder *builder, query *strings.Builder, values *[]interface{},
) {
	*values = append(*values, v.value)
	query.WriteString(fmt.Sprintf("$%d", len(*values)))
}

type cmpKind int

const (
	eqCmp cmpKind = iota
	notEqCmp
	lessCmp
	greaterCmp
	lessEqualCmp
	greaterEqualCmp
)

type cmp struct {
	kind     cmpKind
	lhs, rhs Value
}

func (c cmp) Or(o BoolExpr) BoolExpr {
	return binaryExpr{kind: orExpr, lhs: c, rhs: o}
}

func (c cmp) And(o BoolExpr) BoolExpr {
	return binaryExpr{kind: andExpr, lhs: c, rhs: o}
}

func (c cmp) build(
	builder *builder, query *strings.Builder, values *[]interface{},
) {
	c.lhs.build(builder, query, values)
	switch c.kind {
	case eqCmp:
		if val, ok := c.rhs.(value); ok && val.value == nil {
			query.WriteString(" IS NULL")
			return
		}
		query.WriteString(" = ")
	case notEqCmp:
		if val, ok := c.rhs.(value); ok && val.value == nil {
			query.WriteString(" IS NOT NULL")
			return
		}
		query.WriteString(" <> ")
	case lessCmp:
		query.WriteString(" < ")
	case greaterCmp:
		query.WriteString(" > ")
	case lessEqualCmp:
		query.WriteString(" <= ")
	case greaterEqualCmp:
		query.WriteString(" >= ")
	default:
		panic(fmt.Errorf("unsupported expr %d", c.kind))
	}
	c.rhs.build(builder, query, values)
}

type exprKind int

const (
	orExpr exprKind = iota
	andExpr
)

type binaryExpr struct {
	kind     exprKind
	lhs, rhs BoolExpr
}

func (e binaryExpr) Or(o BoolExpr) BoolExpr {
	return binaryExpr{kind: orExpr, lhs: e, rhs: o}
}

func (e binaryExpr) And(o BoolExpr) BoolExpr {
	return binaryExpr{kind: andExpr, lhs: e, rhs: o}
}

func (e binaryExpr) build(
	builder *builder, query *strings.Builder, values *[]interface{},
) {
	e.lhs.build(builder, query, values)
	switch e.kind {
	case orExpr:
		query.WriteString(" OR ")
	case andExpr:
		query.WriteString(" AND ")
	default:
		panic(fmt.Errorf("unsupported expr %d", e.kind))
	}
	e.rhs.build(builder, query, values)
}
