package logbroker

import (
	"context"
	"fmt"

	zp "go.uber.org/zap"
	"go.uber.org/zap/zapcore"

	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue"
	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue/log/corelogadapter"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/ydb/auth/tvm2"
)

type Reader interface {
	C() <-chan persqueue.Event
	Close() error
}

type reader struct {
	readers   []persqueue.Reader
	context   context.Context
	cancel    context.CancelFunc
	channel   chan persqueue.Event
	endpoints []string
	topic     string
	consumer  string
	logger    log.Logger
	auth      ydb.Credentials
}

func (r *reader) C() <-chan persqueue.Event {
	return r.channel
}

func (r *reader) Close() error {
	r.cancel()
	for _, reader := range r.readers {
		<-reader.Closed()
	}
	var lastErr error
	for _, reader := range r.readers {
		if err := reader.Err(); err != nil && err != context.Canceled {
			lastErr = err
		}
	}
	return lastErr
}

type Option func(*reader) error

func WithConsumer(consumer string) Option {
	return func(r *reader) error {
		r.consumer = consumer
		return nil
	}
}

func WithTopic(topic string) Option {
	return func(r *reader) error {
		r.topic = topic
		return nil
	}
}

func WithEndpoint(endpoints ...string) Option {
	return func(r *reader) error {
		r.endpoints = endpoints
		return nil
	}
}

func WithProduction(r *reader) error {
	r.endpoints = []string{
		"sas.logbroker.yandex.net",
		"vla.logbroker.yandex.net",
		"iva.logbroker.yandex.net",
	}
	return nil
}

func WithTVM(tvm tvm.Client, alias string) Option {
	return func(r *reader) (err error) {
		r.auth, err = tvm2.NewTvmCredentialsForAlias(
			r.context, tvm, alias, r.logger,
		)
		return
	}
}

func WithLogger(logger log.Logger) Option {
	return func(r *reader) (err error) {
		r.logger = logger
		return
	}
}

func NewReader(options ...Option) (Reader, error) {
	logger := zap.Must(zp.Config{
		Level:            zp.NewAtomicLevelAt(zp.InfoLevel),
		Encoding:         "console",
		OutputPaths:      []string{"stdout"},
		ErrorOutputPaths: []string{"stderr"},
		EncoderConfig: zapcore.EncoderConfig{
			MessageKey:     "msg",
			LevelKey:       "level",
			TimeKey:        "ts",
			CallerKey:      "caller",
			EncodeLevel:    zapcore.CapitalColorLevelEncoder,
			EncodeTime:     zapcore.ISO8601TimeEncoder,
			EncodeDuration: zapcore.StringDurationEncoder,
			EncodeCaller:   zapcore.ShortCallerEncoder,
		},
	})
	r := reader{
		channel: make(chan persqueue.Event),
		logger:  logger,
	}
	r.context, r.cancel = context.WithCancel(context.Background())
	for _, option := range options {
		if err := option(&r); err != nil {
			return nil, err
		}
	}
	if len(r.endpoints) == 0 {
		return nil, fmt.Errorf("endpoints must be specified")
	}
	if r.consumer == "" {
		return nil, fmt.Errorf("consumer must be specified")
	}
	if r.topic == "" {
		return nil, fmt.Errorf("topic must be specified")
	}
	if r.auth == nil {
		return nil, fmt.Errorf("auth provider must be specified")
	}
	for _, endpoint := range r.endpoints {
		endpoint := endpoint
		c := persqueue.NewReader(persqueue.ReaderOptions{
			Endpoint:       endpoint,
			Credentials:    r.auth,
			Consumer:       r.consumer,
			Topics:         []persqueue.TopicInfo{{Topic: r.topic}},
			RetryOnFailure: true,
			ReadOnlyLocal:  true,
			MaxReadSize:    128 * 1024 * 1024,
			Logger:         corelogadapter.New(r.logger),
		})
		r.readers = append(r.readers, c)
		go func() {
			for {
				select {
				case <-r.context.Done():
					return
				default:
				}
				if _, err := c.Start(r.context); err != nil {
					r.logger.Warn(
						"Unable to start reader",
						log.String("endpoint", endpoint),
						log.Error(err),
					)
					continue
				}
				break
			}
			for {
				select {
				case message := <-c.C():
					select {
					case r.channel <- message:
					case <-r.context.Done():
						return
					}
				case <-r.context.Done():
					return
				}
			}
		}()
	}
	return &r, nil
}
