package secret

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"os"
	"strings"
)

// Secret represents secret value.
type Secret struct {
	raw   json.RawMessage
	value string
}

// FromString creates secret from string.
func FromString(str string) Secret {
	raw, err := json.Marshal(str)
	if err != nil {
		panic(err)
	}
	return Secret{raw: raw, value: str}
}

// MarshalJSON returns JSON representation of secret.
func (s Secret) MarshalJSON() ([]byte, error) {
	return s.raw, nil
}

// UnmarshalJSON parses secret from JSON.
func (s *Secret) UnmarshalJSON(data []byte) error {
	var value string
	if err := json.Unmarshal(data, &value); err != nil {
		var value struct {
			Type string `json:"type"`
			Name string `json:"name"`
		}
		if err := json.Unmarshal(data, &value); err != nil {
			return err
		}
		switch value.Type {
		case "env", "environ":
			val, ok := os.LookupEnv(value.Name)
			if !ok {
				return fmt.Errorf(
					"environ variable %q does not exists", value.Name,
				)
			}
			s.value, s.raw = strings.TrimSpace(val), data
		case "file":
			val, err := ioutil.ReadFile(value.Name)
			if err != nil {
				return fmt.Errorf(
					"unable to read %q file: %v", value.Name, err,
				)
			}
			s.value, s.raw = strings.TrimSpace(string(val)), data
		default:
			return fmt.Errorf("unsupported %q type", value.Type)
		}
		return nil
	}
	s.value, s.raw = value, data
	return nil
}

// Secret returns secret value.
func (s Secret) Secret() string {
	return s.value
}
