package solomon

import (
	"bytes"
	"context"
	"encoding/json"
	"io/ioutil"
	"log"
	"math"
	"net/http"
	"net/url"
	"sort"
	"strings"
	"sync"
	"time"

	"a.yandex-team.ru/drive/library/go/auth"
)

type timedValue struct {
	time  int64
	value float64
}

type signalValue struct {
	*Signal
	timedValue
}

type Client struct {
	project  string
	cluster  string
	service  string
	endpoint string
	context  context.Context
	cancel   context.CancelFunc
	waiter   sync.WaitGroup
	auth     auth.Provider
	tags     map[string]string
	signals  map[string]*Signal
	mutex    sync.Mutex
	values   []signalValue
	client   http.Client
}

type Signal struct {
	timedValue
	updated bool
	name    string
	tags    map[string]string
	values  []timedValue
	mutex   sync.Mutex
}

var StubSignal *Signal

func (s *Signal) buildValueUnsafe(ts int64) {
	if ts > s.time {
		if s.updated {
			s.values = append(s.values, s.timedValue)
			s.value = 0
			s.updated = false
		}
		s.time = ts
	}
}

func (s *Signal) Add(value float64) {
	if s == nil {
		return
	}
	ts := time.Now().Unix()
	s.mutex.Lock()
	defer s.mutex.Unlock()
	s.buildValueUnsafe(ts)
	s.value += value
	s.updated = true
}

func (s *Signal) Set(value float64) {
	if s == nil {
		return
	}
	ts := time.Now().Unix()
	s.mutex.Lock()
	defer s.mutex.Unlock()
	s.buildValueUnsafe(ts)
	s.value = value
	s.updated = true
}

func getSignalKey(name string, tags map[string]string) string {
	var parts []string
	for key, value := range tags {
		parts = append(parts, key+":"+value)
	}
	sort.Strings(parts)
	var fullName strings.Builder
	fullName.WriteString(name)
	for _, part := range parts {
		fullName.WriteRune('~')
		fullName.WriteString(part)
	}
	return fullName.String()
}

func (c *Client) Signal(name string, tags map[string]string) *Signal {
	c.mutex.Lock()
	defer c.mutex.Unlock()
	key := getSignalKey(name, tags)
	signal, ok := c.signals[key]
	if ok {
		return signal
	}
	signal = &Signal{
		name:       name,
		tags:       tags,
		timedValue: timedValue{time: math.MinInt64},
	}
	c.signals[key] = signal
	return signal
}

func (c *Client) Close() {
	c.cancel()
	c.waiter.Wait()
}

func (c *Client) loop() {
	defer c.waiter.Done()
	ticker := time.NewTicker(15 * time.Second)
	defer ticker.Stop()
	for {
		select {
		case <-c.context.Done():
			c.push(time.Now().Unix() + 5)
			return
		case <-ticker.C:
			c.push(time.Now().Unix() - 5)
		}
	}
}

func (c *Client) buildPayload() ([]byte, error) {
	type sensor struct {
		Labels map[string]string `json:"labels"`
		TS     string            `json:"ts"`
		Value  interface{}       `json:"value"`
	}
	var payload struct {
		Sensors []sensor `json:"sensors"`
	}
	for _, value := range c.values {
		signal := sensor{
			Labels: map[string]string{},
			Value:  value.value,
			TS:     time.Unix(value.time, 0).UTC().Format("2006-01-02T15:04:05Z"),
		}
		for key, value := range c.tags {
			signal.Labels[key] = value
		}
		for key, value := range value.Signal.tags {
			signal.Labels[key] = value
		}
		signal.Labels["sensor"] = value.Signal.name
		payload.Sensors = append(payload.Sensors, signal)
	}
	return json.Marshal(payload)
}

func (c *Client) push(ts int64) {
	for _, signal := range c.signals {
		var values []timedValue
		func() {
			signal.mutex.Lock()
			defer signal.mutex.Unlock()
			signal.buildValueUnsafe(ts)
			values = signal.values
			signal.values = nil
		}()
		for _, value := range values {
			c.values = append(c.values, signalValue{
				Signal:     signal,
				timedValue: value,
			})
		}
	}
	if len(c.values) == 0 {
		return
	}
	data, err := c.buildPayload()
	if err != nil {
		log.Println("Error:", err)
		return
	}
	req, err := http.NewRequest(
		http.MethodPost,
		c.endpoint+"/push",
		bytes.NewBuffer(data),
	)
	if err != nil {
		log.Println("Error:", err)
		return
	}
	req.Header.Add("Content-Type", "application/json")
	query := url.Values{}
	query.Set("project", c.project)
	query.Set("cluster", c.cluster)
	query.Set("service", c.service)
	req.URL.RawQuery = query.Encode()
	if c.auth != nil {
		if err := c.auth.UpdateRequest(req); err != nil {
			log.Println("Error:", err)
			return
		}
	}
	resp, err := c.client.Do(req)
	if err != nil {
		log.Println("Error:", err)
		return
	}
	defer func() {
		if err := resp.Body.Close(); err != nil {
			log.Println("Error:", err)
		}
	}()
	body, _ := ioutil.ReadAll(resp.Body)
	if resp.StatusCode != http.StatusOK {
		log.Printf("Status %v with body: %v", resp.Status, string(body))
		return
	}
	c.values = nil
}

type Option func(*Client) error

func WithAuth(auth auth.Provider) Option {
	return func(client *Client) error {
		client.auth = auth
		return nil
	}
}

func WithEndpoint(endpoint string) Option {
	return func(client *Client) error {
		switch endpoint {
		case "", "production":
			client.endpoint = ProductionEndpoint
		default:
			client.endpoint = endpoint
		}
		return nil
	}
}

func WithTags(tags map[string]string) Option {
	return func(client *Client) error {
		client.tags = tags
		return nil
	}
}

func NewClient(
	project, cluster, service string, options ...Option,
) (*Client, error) {
	context, cancel := context.WithCancel(context.Background())
	client := Client{
		project:  project,
		cluster:  cluster,
		service:  service,
		endpoint: ProductionEndpoint,
		context:  context,
		cancel:   cancel,
		signals:  map[string]*Signal{},
	}
	for _, option := range options {
		if err := option(&client); err != nil {
			return nil, err
		}
	}
	client.waiter.Add(1)
	go client.loop()
	return &client, nil
}
